var EXPORTED_SYMBOLS = ["Listquery"];

// Translates a fullquery into a linear list of term-plus-operator elements

/* TODO - handle operator precedences
Some websites (newsbank) have a list of pulldowns for ops, but
don't take them left-to-right. instead 'or' has a higher precedence,
so that A or B and C meand A or (B and C). This breaks the basic
assumption of linear queries.
Yet it ought to be possible to manage such.
 Plan a: Add a chackbox to disable mixed operators.
 Plan b: Add a set of radio buttons to choose between different setups
  - Left-to-right
  - OR is king
  - AND is king

How to handle "OR is king"
 Plan 1: If the root of the query tree is not 'or', refuse with "query
 too complex.
 Plan 2: Rearrange the query tree so that the root node is OR.
 For example, ( A or B ) and C has the 'and' as the root. But it can be
 rewritten as ( A and C ) or ( B and C ), which can then be processed
 into a linear thing.

*/

/* TODO - Add mappings for field names and operators
(can not get the bug number, network is down)
Charles suggested adding mappings for field names and operator names,
so that those will not have to be mapped individually.
We have 'field' and 'op', so let's call the mapped values 'fieldname' and
'opname'.
  - Add a tab for the mappings
  - Do the actual mapping
  - Add capabilityFlagDefault for the mapped fields (and for default field)
*/



Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');

Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/flatquery.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/queryHelper.js');

var logger = logging.getLogger();

// Constructor
var Listquery = function () {
  this.conf = {};
  this.conf['in'] =  { path:'$.input', key:'fullquery' };
  this.conf['out'] = { path:'$.temp',  key:'listquery' };
  this.conf['maxterms'] = "";
  this.conf['defaultindex'] = "keyword";
  this.conf['makestringterm'] = true;
  this.conf['trunc_left'] = false;
  this.conf['trunc_right'] = true;
  this.conf['trunc_both'] = false;
  this.conf['trunc_mask'] = false;
  this.conf['trunc_wildcard'] = "*";
  this.conf['quote_phrase_L'] = '"';
  this.conf['quote_phrase_R'] = '"';
  this.conf['quote_word_L'] = '';
  this.conf['quote_word_R'] = '';

};

Listquery.prototype = new Step();
Listquery.prototype.constructor = Listquery;
Listquery.prototype.init = function() {};



Listquery.prototype.draw = function(surface,win) {
  var vb = xmlHelper.appendNode(surface, "vbox" );
  //xulHelper.jsonPathMapField(vb, this, "in", "out");
  xulHelper.jsonPathField(vb, this, this.conf, "in", "Source: ",
                          { path:'$.input', key:'fullquery' },
                          { rwmode:"r", containermode:"anynode" } );
  xulHelper.jsonPathField(vb, this, this.conf, "out", "Output: ", 
                          { path:'$.input', key:'fullquery' },
                          { rwmode:"w", containermode:"anynode" });
  var botb = xmlHelper.appendNode(vb, "hbox" );
  var leftb = xmlHelper.appendNode(botb, "vbox" );
  xulHelper.inputField(leftb,this, "maxterms", "Max terms:", 220, {width:80} );
  xulHelper.inputField(leftb,this, "defaultindex", "If the query does not specify "+
     "an index, default to", 220, {width:80} );
  xmlHelper.appendNode(botb, "separator", null,
                       {orient:"vertical", class:"groove"} );

  var rightb = xmlHelper.appendNode(botb, "vbox" );
  var makecheck = xulHelper.checkbox(rightb, this, "makestringterm",
                     "Make a 'stringterm' for each term" );

  var detailb = xmlHelper.appendNode(rightb, "hbox" );
  var truncb =  xmlHelper.appendNode(detailb, "vbox" );
  xulHelper.checkbox(truncb, this, "trunc_right", "Right truncation (watermel*)");
  xulHelper.checkbox(truncb, this, "trunc_left", "Left truncation (*termelon)");
  xulHelper.checkbox(truncb, this, "trunc_both", "Both truncation (*termel*)");
  xulHelper.checkbox(truncb, this, "trunc_mask", "Masking (wat*lon)");
  xmlHelper.appendNode(detailb, "separator", null,
                       {orient:"vertical", class:"groove"} );

  var quoteb =  xmlHelper.appendNode(detailb, "vbox" );
  xulHelper.inputField(quoteb,this, "trunc_wildcard", "Trunc wildcard", 120,
                       {width:30}  );
  var phraseb = xmlHelper.appendNode(quoteb, "hbox" );
  xulHelper.inputField(phraseb,this, "quote_phrase_L", "Phrase quoting", 120,
                       {width:30}  );
  xulHelper.inputField(phraseb,this, "quote_phrase_R", "", 5,
                       {width:30}  );
  var wordb = xmlHelper.appendNode(quoteb, "hbox" );
  xulHelper.inputField(wordb,this, "quote_word_L", "Word quoting", 120,
                       {width:30}  );
  xulHelper.inputField(wordb,this, "quote_word_R", "", 5,
                       {width:30}  );

  // Disable the stringterm inputs if we don't make stringterms
  detailb.hidden = ! makecheck.checked;

  makecheck.addEventListener("command", function(e) {
    detailb.hidden = ! makecheck.checked;
    }, false);

};

Listquery.prototype.run = function (task) {
  var fq = jsonPathHelper.getFirst(this.conf.in, task.data);
  if ( Array.isArray(fq) )
      fq = fq[0];
  var maxt = this.conf['maxterms'];
  task.debug("Listquery: max='" + maxt + "' f=" + fq );

  // TODO - Parameters for input and output, maxterms, etc.
  //var fq = task.data.input.fullquery ;
  var flat = FlatQuery.flattenfq(fq, maxt, this.conf.defaultindex);
  if ( this.conf.makestringterm && flat ) {
    for ( var i = 0; i < flat.length; i++ ) {
      var truncterm = queryHelper.trunc_term( flat[i], this.conf.trunc_wildcard,
          this.conf.trunc_left, this.conf.trunc_right,
          this.conf.trunc_both, this.conf.trunc_mask );
      var quoted = queryHelper.quote_string(flat[i],truncterm,
          true, this.conf.quote_phrase_L, this.conf.quote_phrase_R,
          true, this.conf.quote_word_L, this.conf.quote_word_R );
          // unspecified structure will be quoted as a word, by default
      flat[i].stringterm = quoted;
    }
  }
  task.debug("Lq: flat= " + JSON.stringify(flat) );
  jsonPathHelper.set(this.conf.out, flat, task.data);

  /*
  // debug: Can we get it out again
  task.debug("Lq: out= " + JSON.stringify(this.conf.out) );
  task.debug("Lq: flat= " + JSON.stringify(flat) );
  var f2 = jsonPathHelper.get(this.conf.out, task.data);
  task.debug("Lq: refetched flat: " + typeof(f2) + " " + JSON.stringify(f2) );
  var j1 = jsonPath(task.data, this.conf.out.path);
  task.debug("Lq: j1 = " + typeof(j1) + " " + JSON.stringify(j1) );
  var j2 = j1[0]["listquery"];
  task.debug("Lq: j2 = " + typeof(j2) + " " + JSON.stringify(j2) );
*/
};


///////////////////////////
// Housekeeping

Listquery.prototype.getClassName = function () {
  return "Listquery";
};

Listquery.prototype.getDisplayName = function () {
  return "Listquery";
};

Listquery.prototype.getDescription = function () {
  return "Translate a complex query into a linear list of terms";
};

Listquery.prototype.getVersion = function () {
  //return "1.0";
  return "1.1";  // Added stringterm
};

// Not much point in a renderArgs, way too complex to show in the builder
Listquery.prototype.renderArgs = function () {
  return "";
};

Listquery.prototype.capabilityFlagDefault = function ( flag ) {
  //dump("checking query-and, conf = " + JSON.stringify(this.conf) + "\n");  
  var used = this.getUsedArgs();
  for ( var a in used ) {
    if ( flag == "index-" + a )
      return true;
  }
  if ( flag == "index-" + this.conf['defaultindex'] )
    return true; // we assume the default index will be used for something
  if ( flag == "query-full" )
    return true; // Listquery is kind of fullquery support
  return null;
};

Listquery.prototype.getUsedArgs = function () {
  if (this.conf.in && this.conf.in.path === "$.input")
    return [this.conf.in.key];
  else
    return [];
};

Listquery.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if ( confVer == "1.0" ) {
    conf['makestringterm'] = false;
    conf['trunc_left'] = false;
    conf['trunc_right'] = true;
    conf['trunc_both'] = false;
    conf['trunc_mask'] = false;
    conf['trunc_wildcard'] = "*";
    conf['quote_phrase_L'] = '"';
    conf['quote_phrase_R'] = '"';
    conf['quote_word_L'] = '';
    conf['quote_word_R'] = '';
  }
  return true;
};
