var EXPORTED_SYMBOLS = ["Simplequery"];

// Takes a fullquery and extracts the most simple search terms
// from it, like the old cf-zserver used to do.

// TODO - UI: If ranges supported, do not allow selecting phrase/trunc

// TODO - Defaults from the template
//          - List input fields in the template
//             - we will drop them from inputs later
//          - Only reasonable search fields (no temp, fullquery, etc)
//          - Tell which ones can do the start/end trick (year)
//          - Good default settings when starting a new step

Components.utils.import('resource://indexdata/runtime/Connector.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/queryHelper.js');
Components.utils.import('resource://indexdata/util/flatquery.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');

var logger = logging.getLogger();


// Constructor
var Simplequery = function () {
  this.conf = {};
  this.conf['in'] =  { path:'$.input', key:'fullquery' };
  //this.conf['out'] =  { path:'$.input', key:'' };
    // The helper can't handle that very well...
  this.conf['outpath'] =  '$.temp' ;
  this.conf['truncchar'] = "*";
  this.conf['phrasequote_L'] = '"';
  this.conf['phrasequote_R'] = '"';
  this.conf['defaultindex'] = ""; // TODO - get from template to 'keyword'
  this.conf['indexes'] = []; // list of index names
  // For each field, something like:
  // keyword_trunc_L, _R, _LR, _Mask,  all true/false
  // keyword_phrase, keyword_ranges, also true/false
};


Simplequery.prototype = new Step();
Simplequery.prototype.constructor = Simplequery;
Simplequery.prototype.init = function() {};


///////////////////
// UI


Simplequery.prototype.draw = function(surface,win) {
  var tabs = xulHelper.tabBox( surface,
         [ "General", "Fields" ], { flex: 1 } );
  this.drawGeneralTab( tabs[0], this);
  this.drawFieldsTab( tabs[1], this);

};

Simplequery.prototype.drawGeneralTab = function (surface, context) {
  var vb = xmlHelper.appendNode(surface, "vbox", null, { flex: 1, pack:"start" } );
  //vb.style.backgroundColor = "blue";
  var jsonin = xulHelper.jsonPathField(vb, context, "in", "In " );
  //jsonin.parentNode.align="top";
  //jsonin.parentNode.style.backgroundColor = "green";
  var outp = xulHelper.selectField(vb, context, "outpath", "Out",
                        { "$.temp":"temp", "$.input":"input" } );
  xulHelper.inputField(vb, context, "truncchar", "Truncation wildcard",
                       200, {width:30});

  var pb = xmlHelper.appendNode(vb, "hbox" );
  //xulHelper.captionField(pb, "Phrase quoting",  { width: 200 });
  //xulHelper.arraySelectField(pb, context, 'phrasequote', phrasequotes);

  xulHelper.inputField(pb, context, "phrasequote_L",
                       "Phrase quoting", 200, {width:30});
  xulHelper.inputField(pb, context, "phrasequote_R", "", 0, {width:30});

  //xmlHelper.appendNode(vb, "spacer", null, { flex:9 } );
    // TODO: This should not be necessary, just trying to keep things at top
    // of panel
};

// Helper to set a default for a index attribute
// If the setting is not in conf, tries to take from the default field. If
// even that is not there, uses the defaultvalue given
function fielddef( context, fieldname, suffix, defaultvalue) {
  var ci = fieldname + suffix;
  if ( context.conf[ci] === undefined ) {
    var di = context.conf.defaultindex + suffix;
    if ( context.conf[di] === undefined )
      context.conf[ci] = defaultvalue;
    else
      context.conf[ci] = context.conf[di];
  }
}; // fielddef

// Helper to draw a field line in the fields tab
function addline(surface, context, indexname) {
  var frow = xmlHelper.appendNode(surface, "row", null, { flex:1 } );

  //logger.debug("addline: conf: " + JSON.stringify(context.conf) );
  var fi = context.conf.indexes.indexOf(indexname);
  if ( context.conf.indexes.length == 0 ||
       context.conf.defaultindex == "" ) {
    context.conf.defaultindex = indexname;
  }
  if ( fi == -1 ) {
    context.conf.indexes.push(indexname);
    // TODO - get defaults for the options
  }
  var dispname = indexname;
  if ( context.conf.defaultindex == indexname )
    dispname += " (*)";
  var fb = xmlHelper.appendNode(frow, "hbox", null, {align:"center"} );
  xulHelper.captionField(fb, dispname,  { width: 140 });

  fb = xmlHelper.appendNode(frow, "hbox", null, {flex:1} );
  fielddef( context, indexname, "_trunc_L", false );
  xulHelper.checkbox(fb, context, indexname+"_trunc_L", "L")
  fielddef( context, indexname, "_trunc_R", true );
  xulHelper.checkbox(fb, context, indexname+"_trunc_R", "R")
  fielddef( context, indexname, "_trunc_LR", false );
  xulHelper.checkbox(fb, context, indexname+"_trunc_LR", "L+R")
  fielddef( context, indexname, "_trunc_Mask", false );
  xulHelper.checkbox(fb, context, indexname+"_trunc_Mask", "Mask")

  fb = xmlHelper.appendNode(frow, "hbox" );
  fielddef( context, indexname, "_phrase", false );
  xulHelper.checkbox(fb, context, indexname+"_phrase", "Supported")


  fb = xmlHelper.appendNode(frow, "hbox" );
  fielddef( context, indexname, "_ranges", false );
  xulHelper.checkbox(fb, context, indexname+"_ranges", "Start/End")
  // TODO - if ranges are supported, disable (and unset) trunc/phrase!

  fb = xmlHelper.appendNode(frow, "hbox" );
  var rmbut = xmlHelper.appendNode(fb, "image", null,
      {src: "chrome://cfbuilder/content/icons/window-close.png"}, null);
  rmbut.addEventListener("click", function(e) {
    xmlHelper.removeNode(frow);
    var fi = context.conf.indexes.indexOf(indexname);
    if ( fi != -1 )
      context.conf.indexes.splice(fi,1);
    delete context.conf[indexname+"_trunc_L"];
    delete context.conf[indexname+"_trunc_R"];
    delete context.conf[indexname+"_trunc_LR"];
    delete context.conf[indexname+"_trunc_Mask"];
    delete context.conf[indexname+"_phrase"];
    if ( indexname == context.conf.defaultindex &&
         context.conf.indexes.length > 0 ) { // set a new default
      context.conf.defaultindex = context.conf.indexes[0];
    redrawfields(surface, context);

    }
  }, false);

}

// Helper to (re)draw all field lines
function redrawfields(surface,context) {
  xmlHelper.emptyChildren(surface);
  var frow = xmlHelper.appendNode(surface, "row", null,{ flex: 1 } );

  xulHelper.captionField(frow, "Field" );
  xulHelper.captionField(frow, "Truncation", {flex:1} );
  xulHelper.captionField(frow, "Phrases");
  xulHelper.captionField(frow, "Ranges");
  xulHelper.captionField(frow, "(delete)");
  for ( var fn in context.conf.indexes ) {
    addline(surface, context, context.conf.indexes[fn]);
  }
}; // redrawfields

Simplequery.prototype.drawFieldsTab = function (surface, context) {
  var vb = xmlHelper.appendNode(surface, "vbox", null, { flex: 1 } );
  var fieldlist = context.task.getArgNames(); // TODO - Get from template!
//  var fieldsbox = xmlHelper.appendNode(vb, "vbox", null, { flex: 1 } );
  var fieldsgrid = xmlHelper.appendNode(vb, "grid", null, { flex: 1 } );

  // ### Do I need columns here?
  var cols = xmlHelper.appendNode(fieldsgrid, "columns" );
  xmlHelper.appendNode(cols, "column", null, {flex:1} );
  xmlHelper.appendNode(cols, "column", null, {flex:1} );
  xmlHelper.appendNode(cols, "column", null, {flex:1} );
  xmlHelper.appendNode(cols, "column", null, {flex:1} );

  var fieldsbox = xmlHelper.appendNode(fieldsgrid, "rows", null,
                                { flex: 1 } );
  redrawfields(fieldsbox,context);
  xmlHelper.appendNode(vb, "spacer", null, { flex:9 } );
  var addbox = xmlHelper.appendNode(vb, "hbox", null, {align:"center"} );
  xulHelper.captionField(addbox, "Add index" );
  var selectfields = fieldlist; // TODO - Remove duplicates, tmp, etc.
  var addsel = xulHelper.arraySelectField(addbox, context, '',
                     selectfields, ["(choose)"] );
  addsel.addEventListener("command", function(e) {
    var fn = addsel.value;
    context.task.debug("command: " + fn );
    addline(fieldsbox,context,fn);
    addsel.selectedIndex = 0; // re-selected the '(choose)'
  }, false );
};


/////////////////////////
// Run

// Check if we have a (year?) range
Simplequery.prototype.check_ranges = function ( n, field, qvalues ) {
  if ( !this.conf[ field + '_ranges' ] )
    return false; // does not support ranges. Proceed as usual
  queryHelper.validate_no_trunc(n,
         "Truncation not supported with ranges");
  queryHelper.validate_no_struct(n,
         "Structure '" + n.structure + "' not supported with ranges");
  var startfield = "start" + field;  // "startyear", like in old cf-zserver
  var endfield = "end" + field;
  var endpoints = queryHelper.new_endpoints(
            qvalues[startfield], qvalues[endfield]);
  queryHelper.range_endpoint(n, endpoints);
  if ( endpoints.start)
    qvalues[startfield] = endpoints.start;
  if ( endpoints.end )
    qvalues[endfield] = endpoints.end;
  if ( endpoints.start && ( endpoints.start == endpoints.end ) )
    qvalues[field] = endpoints.start; // one year specified
  return true;
},

// Process one term. Appends into proper qvalues string
Simplequery.prototype.term = function ( n, qvalues ) {
  var field = n.field;
  var term = n.term;
  if ( field == undefined )
    field = this.conf.defaultindex;
  if ( this.conf.indexes.indexOf(field) == -1 )
    throw new StepError("Field " + field + " not supported");
  queryHelper.validate_position(n);
  queryHelper.validate_completeness(n);
  if ( this.check_ranges(n, field, qvalues) ) {
    return; // checkranges sets it values, and throws its errors
    // if there was a range, we don't support trunc/phrase stuff
  }
  queryHelper.validate_no_relation(n,
    "Relation '" + n.relation + "' not supported for " + field );
  var term = queryHelper.trunc_term( n,  this.conf.truncchar,
         this.conf[ field+"_trunc_L" ], this.conf[ field+"_trunc_R" ],
         this.conf[ field+"_trunc_LR" ], this.conf[ field+"_trunc_Mask" ] );
  logger.info ( "sq:term: " + field + ": " +
      this.conf[ field + "_phrase" ] + " : " +
      this.conf['phrasequote_L'] + " / " + this.conf['phrasequote_R'] );
  var quotedterm = queryHelper.quote_string(n, term,
      this.conf[ field + "_phrase" ],
      this.conf['phrasequote_L'], this.conf['phrasequote_R'] );
        // further parameters omitted, default to accepting but not quoting

  if ( qvalues[field] )
    qvalues[field] += " "
  else
    qvalues[field] = "";
  qvalues[field] += quotedterm;
}; // term


Simplequery.prototype.node = function ( n, qvalues ) {
  if ( n.term == undefined ) {
    if ( n.op != 'and' )
      throw new StepError("only AND supported" );
    this.node( n.s1, qvalues );
    this.node( n.s2, qvalues );
  } else { // must be a term node
    this.term( n, qvalues );
  }
}; // node

Simplequery.prototype.run = function (task) {
  var fq = jsonPathHelper.get(this.conf.in, task.data);
  task.debug("sq.run: first fq:" + fq + " t=" + typeof(fq) +
        " json: '" + JSON.stringify(fq) + "'");
  if ( Array.isArray(fq) && fq.length > 0 ) {
    task.debug("sq: Got an array of " + fq.length + " elements, taking the [0]");
    fq = fq[0];
  }

  if ( Array.isArray(fq) && fq[0] &&
    ( typeof(fq[0].op) != "string" ||
      typeof(fq[0].term) != "string" )  ) {
      // Only a listquery can have both op and term!)
    task.debug("sq: Still an array of " + fq.length + " elements, " +
        " that looks like a real fullquery. Taking [0] ");
    fq = fq[0];
  }

  task.debug("sq.run: final fq:" + fq + " t=" + typeof(fq) +
        " json: '" + JSON.stringify(fq) + "'");
  var queryvalues = {};
  // TODO - Accept and process a list query too!
  this.node(fq, queryvalues);
  task.debug("sq: " + JSON.stringify(queryvalues) );
  var out = { path: this.conf.outpath, key:"" };
  for ( var k in queryvalues ) {
    out.key = k;
    jsonPathHelper.set(out, [ queryvalues[k] ], task.data);
      // note, we make an array out of a scalar here!
  }
};

/////////////////////////
// Unit test

Simplequery.prototype.unitTest = function ( ) {
  const defaultsettings = {
    // These apply to all tests, unless overridden by the test itself
    // Things not specified here come from the constructor, or are left
    // undefined.
    outpath: '$.temp',
    defaultindex: "keyword",
    indexes: [ "keyword", "title", "author", "year" ],
  };
  const tests = [
    { 'name': 'single term',
      //'override': {}, // run with the defaults
      'query': '{ "term":"foo" }' ,
      // error: "expected error message",
      'out': {
        'keyword': 'foo',
      },
      capflags: [ "index-keyword", "query-and" ],
      notcapflags: [ "query-or", "index-publisher" ],
    },
    { 'name': 'simple and',
      'query': '{ "op":"and", "s1":{ "term":"foo" }, "s2":{ "term":"bar"}}' ,
      'out': {
        'keyword': 'foo bar',
      },
    },
    { 'name': 'simple or',
      'query': '{ "op":"or", "s1":{ "term":"foo" }, "s2":{ "term":"bar"}}' ,
      error: "only AND supported",
    },
    { 'name': 'ti and au',
      'query': '{ "op":"and", ' +
                 ' "s1":{ "term":"foo", "field":"title" }, ' +
                 ' "s2":{ "term":"bar", "field":"author" } }' ,
      'out': {
        'title': 'foo',
        'author': 'bar',
      },
    },
    { 'name': 'unsupported term',
      'query': '{ "term":"foo", "field":"unsupportedfield" }' ,
      error: "Field unsupportedfield not supported",
    },
    { 'name': 'bad position',
      'query': '{ "term":"foo", "position":"firstinfield" }' ,
      error: "Position not supported",
    },
    { 'name': 'bad completeness',
      'query': '{ "term":"foo", "position":"completesubfield" }' ,
      error: "Position not supported",
    },
    { 'name': 'bad-relation-non-range',
      'query': '{ "term":"foo", "relation":"le" }' ,
      error: "Relation 'le' not supported for keyword",
    },
    { 'name': 'good extras',
      'query': '{ "term":"foo", "position":"any", ' +
                  '"completeness":"incompletesubfield", "relation":"eq" }' ,
      'out': {
        'keyword': 'foo',
      },
    },
    { 'name': 'trunc_L',
      'override': { keyword_trunc_L: true },
      'query': '{ "term":"foo", "truncation":"left" }' ,
      'out': {
        'keyword': '*foo',
      },
      capflags: [ "trunc-left" ],
      notcapflags: [ "trunc-right", "trunc-both", "query-wildcard", "query-phrase" ],
    },
    { 'name': 'trunc_L-bad',
      'override': { keyword_trunc_L: false },
      'query': '{ "term":"foo", "field":"author", "truncation":"left" }' ,
      'error': "Truncation 'left' not supported",
    },
    { 'name': 'trunc_R',
      'override': { keyword_trunc_R: true },
      'query': '{ "term":"foo", "truncation":"right" }' ,
      'out': {
        'keyword': 'foo*',
      },
      capflags: [ "trunc-right" ],
      notcapflags: [ "trunc-left", "trunc-both", "query-wildcard", "query-phrase" ],
    },
    { 'name': 'trunc_R-bad',
      'query': '{ "term":"foo", "field":"author", "truncation":"right" }' ,
      'error': "Truncation 'right' not supported",
    },
    { 'name': 'trunc_LR',
      'override': { keyword_trunc_LR: true },
      'query': '{ "term":"foo", "truncation":"both" }' ,
      'out': {
        'keyword': '*foo*',
      },
      capflags: [ "trunc-both" ],
      notcapflags: [ "trunc-left", "trunc-right", "query-wildcard", "query-phrase" ],
    },
    { 'name': 'trunc_LR-bad',
      'query': '{ "term":"foo", "field":"author", "truncation":"both" }' ,
      'error': "Truncation 'both' not supported",
    },
    { 'name': 'trunc_Mask',
      'override': { keyword_trunc_Mask: true },
      'query': '{ "term":"f?o?o", "truncation":"z39.58" }' ,
      'out': {
        'keyword': 'f*o*o',
      },
    },
    { 'name': 'trunc_Mask-bad',
      'query': '{ "term":"f?o?o", "truncation":"z39.58" }' ,
      'error': "Truncation 'z39.58' not supported",
      capflags: [ "query-wildcard" ],
      notcapflags: [ "trunc-left", "trunc-both", "trunc-right", "query-phrase" ],
    },
    { 'name': 'trunc_Mask- bad single',
      'override': { keyword_trunc_Mask: true },
      'query': '{ "term":"fo#o", "truncation":"z39.58" }' ,
      'error': "Single-character masking with '#' not supported",
    },
    { 'name': 'struct-phrase',
      'override': { keyword_phrase: true },
      'query': '{ "term":"foo", "structure":"phrase" }' ,
      'out': {
        'keyword': '"foo"',
      },
      capflags: [ "query-phrase" ],
      notcapflags: [ "trunc-left", "trunc-both", "trunc-right", "query-wildcard" ],
    },
    { 'name': 'struct-phrase-unsupported',
      'override': { keyword_phrase: false },
      'query': '{ "term":"foo", "structure":"phrase" }' ,
      'error': "Structure 'phrase' not supported"
    },
    { 'name': 'struct-word',
      'override': { keyword_phrase: true },
      'query': '{ "term":"foo", "structure":"word" }' ,
      'out': {
        'keyword': 'foo',
      },
      capflag: [ "index-year" ],
      notcapflags: [ "index-startyear", "index-endyear" ],
    },
    { 'name': 'struct-bad',
      'override': { keyword_phrase: true },
      'query': '{ "term":"foo", "structure":"invalidstructure" }' ,
      'error': "Structure 'invalidstructure' not supported",
    },
    { 'name': 'one-year',
      'override': { year_ranges: true },
      'query': '{ "term":"2005", "field":"year"  }',
      'out': {
        'startyear':'2005',
        'endyear':'2005',
        'year':'2005',
      },
      capflags: [ "index-year", "index-startyear", "index-endyear" ],
    },
    { 'name': 'one-year-eq',
      'override': { year_ranges: true },
      'query': '{ "term":"2005", "field":"year", "relation":"eq"  }',
      'out': {
        'startyear':'2005',
        'endyear':'2005',
        'year':'2005',
      },
      capflags: [ "index-year", "index-startyear", "index-endyear" ],
    },
    { 'name': 'one-year-le',
      'override': { year_ranges: true },
      'query': '{ "term":"2009", "field":"year", "relation":"le"  }',
      'out': {
        'endyear':'2009',
      },
    },
    { 'name': 'one-year-lt',
      'override': { year_ranges: true },
      'query': '{ "term":"2010", "field":"year", "relation":"lt"  }',
      'out': {
        'endyear':'2009',
      },
    },
    { 'name': 'one-year-gt',
      'override': { year_ranges: true },
      'query': '{ "term":"1999", "field":"year", "relation":"gt"  }',
      'out': {
        'startyear':'2000',
      },
    },
    { 'name': 'one-year-ge',
      'override': { year_ranges: true },
      'query': '{ "term":"2000", "field":"year", "relation":"ge"  }',
      'out': {
        'startyear':'2000',
      },
    },
    { 'name': 'year-range-gt-lt',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"1999", "field":"year", "relation":"gt"  },' +
          ' "s2": { "term":"2010", "field":"year", "relation":"lt"  } }',
      'out': {
        'startyear': '2000',
        'endyear': '2009',
      },
    },
    { 'name': 'year-range-ge-le',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"2000", "field":"year", "relation":"ge"  },' +
          ' "s2": { "term":"2009", "field":"year", "relation":"le"  } }',
      'out': {
        'startyear': '2000',
        'endyear': '2009',
      },
    },
    { 'name': 'year-bad-range-ge-ge',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"2000", "field":"year", "relation":"ge"  },' +
          ' "s2": { "term":"2005", "field":"year", "relation":"ge"  } }',
      //'error': "Range with two start values",
      'out': {
        'startyear': '2005',
      },
    },
    { 'name': 'year-bad-range-lt-le',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"2001", "field":"year", "relation":"lt"  },' +
          ' "s2": { "term":"2005", "field":"year", "relation":"le"  } }',
      'out': {
        'endyear': '2000',
      },
    },
    { 'name': 'year-bad-range-ge-eq',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"2000", "field":"year", "relation":"ge"  },' +
          ' "s2": { "term":"2005", "field":"year", "relation":"eq"  } }',
      'out': {
        'startyear': '2005',
        'endyear': '2005',
        'year': '2005',
      },
    },
    { 'name': 'year-bad-range-eq-eq',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"2002", "field":"year", "relation":"eq"  },' +
          ' "s2": { "term":"2005", "field":"year", "relation":"eq"  } }',
      'error': "Impossible range 2005-2002",
    },
    { 'name': 'year-range-eq-eq',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"2005", "field":"year", "relation":"eq"  },' +
          ' "s2": { "term":"2005", "field":"year", "relation":"eq"  } }',
      'out': {
        'startyear': '2005',
        'endyear': '2005',
        'year': '2005',
      },
    },
    { 'name': 'non-numeric-year-range',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"Y2K", "field":"year", "relation":"le"  },' +
          ' "s2": { "term":"Y1K", "field":"year", "relation":"ge"  } }',
      'out': {
        'startyear': 'Y1K',
        'endyear': 'Y2K',
      },
    },
    { 'name': 'two-non-numeric-year-ranges',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          '"s1": { "op":"and", ' +
            ' "s1": { "term":"Y2K", "field":"year", "relation":"le"  },' +
            ' "s2": { "term":"Y1K", "field":"year", "relation":"ge"  } },' +
          '"s2": { "op":"and", ' +
            ' "s1": { "term":"Y3K", "field":"year", "relation":"le"  },' +
            ' "s2": { "term":"Y2K", "field":"year", "relation":"ge"  } } }',
      'out': {
        'startyear': 'Y2K',
        'endyear': 'Y2K',
        'year': 'Y2K',
      },
    },
    { 'name': 'year-range-trunc',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"1999", "field":"year", "relation":"gt", ' +
                   '"truncation":"left" },' +
          ' "s2": { "term":"2010", "field":"year", "relation":"lt"  } }',
      'error': "Truncation not supported with ranges",
    },
    { 'name': 'year-range-phrase',
      'override': { year_ranges: true },
      'query': '{ "op":"and", ' +
          ' "s1": { "term":"1999", "field":"year", "relation":"gt", ' +
                   '"structure":"phrase" },' +
          ' "s2": { "term":"2010", "field":"year", "relation":"lt"  } }',
      'error': "Structure 'phrase' not supported with ranges",
    },
  ];
  logger.info("Starting unit test for Simplequery");
  // fake some runtime
  var connector = new Connector();
  for ( var i=0; i<tests.length; i++) {
    var t=tests[i];
    logger.debug("Test '" + t.name + "' : '" + t.query + "'");
    // for some reason unittest won't show debug output ###
    var task = new Task( connector, "unittest" ); // with empty data
    var aa = new Simplequery;
    aa.task = task;
    for ( var f in defaultsettings ) {
      aa.conf[f] = defaultsettings[f];
    }
    if (t.override)
      for ( var f in t.override ) {
        aa.conf[f] = t.override[f];
      }
    try {
        qry = JSON.parse(t.query);
        jsonPathHelper.set(aa.conf.in, qry, task.data);
    } catch (e) {
      logger.error("" + e + " in '" + t.name +"'" );
      if ( e.fileName )
        logger.error( " in " + e.fileName + ":" + e.lineNumber );
      logger.error(t.query);
      if ( e.fileName )
        logger.error( " in " + e.fileName + ":" + e.lineNumber );
      return false;  // no need to continue further
    }

    var runok = false;
    try {
      aa.run( task );
      runok = true;
    } catch (e) {
      if ( t.error && t.error == e.message ) {
        logger.debug("test " + t.name + " failed as expected");
      } else {
        logger.error( t.name + " FAILED" );
        logger.error( "Exception " + e.message );
        if ( t.error )
          logger.error( "Expected  " + t.error );
        if ( e.fileName )
          logger.error( " in " + e.fileName + ":" + e.lineNumber);
        return false;
      }
    }
    if ( runok ) {
      logger.debug("After test, task.temp = " + JSON.stringify( task.data.temp ) );
      var jp = { path: aa.conf.outpath, key:"" };
      var seen = { };
      for ( var k in t.out ) {
        seen[k] = 1;
        jp.key = k;
        var got = jsonPathHelper.get(jp, task.data);
        logger.debug("Checking " + jp.path + " . "+ k + ": '" + got + "'" );
        if ( got && Array.isArray(got) )
          got = got[0];
        if ( got != t.out[k] ) {
          logger.error( t.name + " FAILED" );
          logger.error( "Different output in " + k + ":" );
          logger.error( "Expected " + t.out[k] );
          logger.error( "But got  " + got );
          return false;
        }
      }
      var op = aa.conf.outpath.replace( /\$\./, "");
      logger.debug("About to cross-check " + op + " " +
         JSON.stringify( task.data[op] ) );
      for ( var k in task.data[op] ) {
        logger.debug("cross-checking " + k );
        if ( ! seen[k] ) {
          logger.error( t.name + " FAILED" );
          logger.error( "Did not expect anything in " + k );
          logger.error( "but got " + task.data[op][k] );
          return false;
        }
      }
      for ( var c in t.capflags ) {
        var f = t.capflags[c];
        logger.debug("Checking cap flag " + c + " " + f );
        if ( !aa.capabilityFlagDefault(f) ) {
          logger.error(t.name + " FAILED");
          logger.error("Capabiltiy flag default " + f + " not set");
          return false;
        }
      }
      for ( var c in t.notcapflags ) {
        var f = t.notcapflags[c];
        logger.debug("Checking cap flag " + c + " " + f );
        if ( aa.capabilityFlagDefault(f) ) {
          logger.error(t.name + " FAILED");
          logger.error("Capabiltiy flag default " + f + " is set");
          return false;
        }
      }
    }
    logger.info( t.name + " OK");
  } // test loop
  logger.debug("All " + tests.length + " tests OK");
  return true;
};

///////////////////////////
// Housekeeping

Simplequery.prototype.getClassName = function () {
  return "Simplequery";
};

Simplequery.prototype.getDisplayName = function () {
  return "Simple query";
};

Simplequery.prototype.getDescription = function () {
  return "Extract simple author/title/etc from a query";
};

Simplequery.prototype.getVersion = function () {
  return "1.0";
};

Simplequery.prototype.renderArgs = function () {
  // TODO - make something that looks nice
  return "";
};

Simplequery.prototype.capabilityFlagDefault = function ( flag ) {
  // TODO - Should this set the fullquery flag? Probably not, this
  // isn't anywhere near 'full'...
  if (flag.substring(0, 6) == "index-") {
    var index = flag.substring(6); // after 'index-'
    if (this.conf.indexes.indexOf(index) != -1 )
      return true;
    // TODO - start/end year stuff  "index-startyear"
    var index = flag.substring(11); // after 'index-start'
    if (flag.substring(0,11) == "index-start" &&
        this.conf.indexes.indexOf(index) != -1 &&
        this.conf[index+"_ranges"] )
      return true;
    var index = flag.substring(9); // after 'index-start'
    if (flag.substring(0,9) == "index-end" &&
        this.conf.indexes.indexOf(index) != -1 &&
        this.conf[index+"_ranges"] )
      return true;
  }
  if ( flag == "query-and" )
    return true;

  // TODO - Do we need to set these. They are only of interest to
  // cf-zserver in non-fq mode, which should not be relevant here
  if ( flag == "trunc-questionmark" && this.conf['truncchar'] == '?' )
    return true;
  if ( flag == "trunc-asterisk" && this.conf['truncchar'] == '*' )
    return true;

  if ( flag == "trunc-left" || flag == "trunc-right" || flag == "trunc-both" ||
       flag == "query-wildcard" || flag == "query-phrase") {  // check if we support such trunc for any
    for( var k in this.conf ) {
      if ( flag == "trunc-left" && k.match(/_trunc_L$/) && this.conf[k] )
        return true;
      if ( flag == "trunc-right" && k.match(/_trunc_R$/) && this.conf[k] )
        return true;
      if ( flag == "trunc-both" && k.match(/_trunc_LR$/) && this.conf[k] )
        return true;
      if ( flag == "query-wildcard" && k.match(/_trunc_Mask$/) && this.conf[k] )
        return true;
      if ( flag == "query-phrase" && k.match(/_phrase$/) && this.conf[k] )
        return true;
    }
  }

};

Simplequery.prototype.getUsedArgs = function () {
  if (this.conf.in && this.conf.in.path === "$.input")
    return [this.conf.in.key];
  else
    return [];
};

Simplequery.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  return true;
};
