/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file cf_engine.cpp
 * \brief Implementation of CF_Embed / CF_Engine.
 */
#include "cf_config.h"
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <assert.h>
#include <unistd.h>
#include <signal.h>
#include <poll.h>
#include <yaz/xmalloc.h>
#include <yaz/snprintf.h>
#include <yaz/nmem.h>
#include <yaz/log.h>

#include "cf_engine.h"
#include "cf_embed.h"

int CF_Embed::write_request(WRBUF w)
{
    ssize_t r = 0;
    size_t off = 0;

    signal(SIGPIPE, SIG_IGN);
    while (off < wrbuf_len(w))
    {
        r = write(request_fd[1], wrbuf_buf(w) + off, wrbuf_len(w) - off);
        if (r == -1)
        {
            if (errno == EINTR)
                continue;
            return -1;
        }
        off += r;
    }
    return 0;
}

void CF_Embed::write_response(const void *buf, size_t count)
{
    size_t r = write(response_fd[1], buf, count);
    if (r != count)
    {
        printf("engine", YLOG_WARN,
               "write_response returned %lld, expected %lld",
               (long long) r, (long long) count);
    }
}

void CF_Embed::write_response_cstr(const char *cstr)
{
    write_response((const void *) cstr, strlen(cstr));
}

CF_Engine::CF_Engine(enum Flags flags,
                     const char *builder_path,
                     const char *app_path,
                     const char *profile_path,
                     const char *cfmain_path,
                     const char *proxy,
                     const char *loglevel,
                     const char *sesname, int capt_fd)
{
    m_priv = 0;
    init(flags, builder_path, app_path, profile_path, cfmain_path, proxy,
         loglevel, sesname, capt_fd);
}

bool CF_Embed::wait_init_handshake(std::string &error_msg)
{
    yaz_timing_t myt = yaz_timing_create();
    yaz_timing_start(myt);

    const char *output = 0;
    bool sys_err = false;
    int wait_init = 20;
    {
        char *str = getenv("CF_WAIT_INIT");
        if (str)
            wait_init = atoi(str);
    }
    bool res = wait_result_parent(&output, wait_init * 1000, &sys_err);
    yaz_timing_stop(myt);

    printf("timing", YLOG_LOG,
           "handshake user=%f real=%f sys=%f",
           yaz_timing_get_user(myt),
           yaz_timing_get_real(myt),
           yaz_timing_get_sys(myt));
    yaz_timing_destroy(&myt);

    if (!res)
    {
        if (output && *output)
        {
            printf("engine", YLOG_LOG,
                   "engine create fail: %s", output);
            error_msg = output;
        }
        else
        {
            error_msg = "engine creation failure";
        }
        return false;
    }
    if (output)
        main_window_xid = atol(output);
    return true;
}

bool CF_Embed::spawn_fork(std::string &error_msg,
                          const char *module_path,
                          const char *app_path, const char *cfmain_path,
                          const char *proxy,
                          int capt_fd)
{
    int i, log_fd = -1;
    FILE *yaz_log_FILE = yaz_log_file();
    if (yaz_log_FILE)
        log_fd = fileno(yaz_log_FILE);

    char *args[24];
    size_t j = 0;
    char fdstr[40];
    NMEM nmem = nmem_create();
    sprintf(fdstr, "%d:%d:%d", request_fd[0], response_fd[1], log_fd);
    args[j++] = nmem_strdup(nmem, "cfmain");
    const char *d = display.get_display();
    if (d)
    {
        args[j++] = nmem_strdup(nmem, "-d");
        args[j++] = nmem_strdup(nmem, d);
    }

    args[j++] = nmem_strdup(nmem, "-f");
    args[j++] = fdstr;
    if (app_path)
    {
        args[j++] = nmem_strdup(nmem, "-a");
        args[j++] = nmem_strdup(nmem, app_path);
    }
    if (profile_path)
    {
        args[j++] = nmem_strdup(nmem, "-r");
        args[j++] = nmem_strdup(nmem, profile_path);
    }
    if (module_path)
    {
        args[j++] = nmem_strdup(nmem, "-m");
        args[j++] = nmem_strdup(nmem, module_path);
    }
    if (proxy)
    {
        args[j++] = nmem_strdup(nmem, "-p");
        args[j++] = nmem_strdup(nmem, proxy);
    }
    if (sesname.length())
    {
        args[j++] = nmem_strdup(nmem, "-s");
        args[j++] = nmem_strdup(nmem, sesname.c_str());
    }
    args[j] = 0;

    if (!cfmain_path || !*cfmain_path)
        cfmain_path = "./cfmain";

    pid = ::fork();
    switch (pid)
    {
    case 0: /* child */
        // close all files except log files and pipe for communication */
        for (i = getdtablesize(); --i > 2; )
        {
            if (i != log_fd && i != capt_fd &&
                i != response_fd[1] && i != request_fd[0])
                close(i);
        }
        signal(SIGTERM, SIG_DFL);
        if (capt_fd != -1)
        {
            close(0);
            dup(capt_fd);
            close(1);
            dup(capt_fd);
            close(2);
            dup(capt_fd);
        }

        execv(cfmain_path, args);

        wrbuf_printf(result_wrbuf, "FAIL exec(%s): %s\n",
                     cfmain_path, strerror(errno));
        write_response_cstr(wrbuf_cstr(result_wrbuf));
        exit(1);
        break;
    case -1: /* error */
        nmem_destroy(nmem);
        throw CF_Engine_Error_System("fork system call failed");
        error_msg = "fork system call failed";
        return false;
    default:  /* parent */
        nmem_destroy(nmem);
        close(response_fd[1]);
        response_fd[1] = -1;
        close(request_fd[0]);
        request_fd[0] = -1;

        printf("engine", YLOG_LOG,
               "CF fork: parentpid=%lld childpid=%lld",
               (long long) getpid(), (long long) pid);

        return wait_init_handshake(error_msg);
    }
    return true;
}

void CF_Engine::init(enum Flags flags,
                     const char *builder_path,
                     const char *app_path, const char *a_profile_path,
                     const char *cfmain_path,
                     const char *proxy, const char *loglevel,
                     const char *sesname, int capt_fd)
{
    if (!builder_path)
	throw CF_Engine_Error_System("builder_path NULL");
    CF_Embed *embed = new CF_Embed(loglevel, sesname, capt_fd);

    if (!a_profile_path)
        a_profile_path = "/tmp/cf_profile_XXXXXX";

    embed->profile_path = xstrdup(a_profile_path);

    if (strstr(embed->profile_path, "XXXXXX"))
    {
        mkdtemp(embed->profile_path);
        embed->profile_temp = 1;
        embed->copy_initial_js_files(embed->profile_path, app_path, loglevel);
    }
    embed->printf("engine", YLOG_LOG,
                  "init: flags=%d, builder_path=%s app_path=%s "
                  "profile_path=%s proxy=%s", flags,
                  builder_path ? builder_path : "null",
                  app_path ? app_path : "null",
                  embed->profile_path ? embed->profile_path : "null",
                  proxy ? proxy : "null");
#if FORCE_DISPLAY
    // hack to spread display to several, already running, Xvfb's
    char display_str[40];

    // :2.0,:3.0,  ..., :150.0
    int no = ((getpid()) % 149U) + 2;

    sprintf(display_str, ":%d.0", no);
    setenv("DISPLAY", display_str, 1);

    embed->logger->printf("engine", "display=%s", display_str);
#endif

    std::string msg;
    bool ret = false;
    if ((flags & 1) == FORK)
    {
        ret = embed->spawn_fork(msg, builder_path, app_path, cfmain_path,
                                proxy, capt_fd);
    }
    if (!ret)
    {
        delete embed;
        throw CF_Engine_Error_System(msg.c_str());
    }
    m_priv = embed;
}

CF_Engine::~CF_Engine()
{
    delete m_priv;
}

void CF_Embed::printf(const char *module, int level, const char *fmt, ...)
{
    va_list ap;
    char *cp, buf[4096];

    va_start(ap, fmt);
    *buf = '\0';
    if (sesname.length())
    {
        strcat(buf, sesname.c_str());
        strcat(buf, " ");
    }
    if (module)
    {
        strcat(buf, module);
        strcat(buf, " ");
    }
    cp = buf + strlen(buf);
    yaz_vsnprintf(cp, sizeof(buf)-100, fmt, ap);

    yaz_log(YLOG_LOG, "%s", buf);
    va_end(ap);
}

void CF_Embed::copy_initial_js_files(const char *a_profile_path,
                                     const char *a_app_path,
                                     const char *loglevel)
{
    WRBUF src_fname = wrbuf_alloc();
    WRBUF dst_fname = wrbuf_alloc();
    const char *fnames[] = {"prefs.js", "user.js", 0};
    int i;
    assert(a_profile_path);

    if (!a_app_path)
        a_app_path = ".";

    for (i = 0; fnames[i]; i++)
    {
        wrbuf_rewind(src_fname);
        wrbuf_rewind(dst_fname);
        wrbuf_printf(src_fname, "%s/defaults/preferences/%s",
                     a_app_path, fnames[i]);
        wrbuf_printf(dst_fname, "%s/%s", a_profile_path, fnames[i]);
        FILE *src_file = fopen(wrbuf_cstr(src_fname), "r");
        if (src_file)
        {
            FILE *dst_file = fopen(wrbuf_cstr(dst_fname), "w");
            if (dst_file)
            {
                while (1)
                {
                    char buf[256];
                    size_t rd = fread(buf, 1, sizeof(buf), src_file);
                    if (rd > 0)
                        fwrite(buf, 1, rd, dst_file);
                    if (rd < sizeof(buf))
                        break;
                }
                if (loglevel && *loglevel)
                    fprintf(dst_file,
                            "user_pref(\"indexdata.cf.logging.root\", \"%s\");\n",
                            loglevel);
                fclose(dst_file);
            }
            fclose(src_file);
        }
    }
    wrbuf_destroy(src_fname);
    wrbuf_destroy(dst_fname);
}

CF_Embed::CF_Embed(const char *a_loglevel, const char *a_sesname, int fd)
    : profile_path(0),
      profile_temp(0),
      pid(0),
      result_wrbuf(0),
      current_task(0),
      main_window_xid(1),
      display(this)
{
    if (a_sesname)
        sesname = a_sesname;

    if (!display.lock(fd))
    {
        const char *e = display.get_error();

        printf("engine", YLOG_FATAL, "display lock: %s", e);

        throw CF_Engine_Error_System(e);
    }
    if (pipe(request_fd) < 0)
    {
        throw CF_Engine_Error_System("pipe system call failed");
    }
    if (pipe(response_fd) < 0)
    {
        close(request_fd[0]);
        close(request_fd[1]);
        throw CF_Engine_Error_System("pipe system call failed");
    }
    result_wrbuf = wrbuf_alloc();
    current_task = wrbuf_alloc();
}

CF_Embed::~CF_Embed()
{
    const char *output = 0;
    WRBUF w = wrbuf_alloc();
    wrbuf_puts(w, "quit\n");
    write_request(w);
    wrbuf_destroy(w);

    bool sys_err = false;
    int wait_init = 20;
    {
        char *str = getenv("CF_WAIT_INIT");
        if (str)
            wait_init = atoi(str);
    }
    wait_result_parent(&output, wait_init * 1000, &sys_err);

    if (request_fd[0] != -1)
        close(request_fd[0]);
    if (request_fd[1] != -1)
        close(request_fd[1]);
    if (response_fd[0] != -1)
        close(response_fd[0]);
    if (response_fd[1] != -1)
        close(response_fd[1]);
    wrbuf_destroy(result_wrbuf);

    if (pid)
    {
        printf("engine", YLOG_LOG, "kill %lld", (long long) pid);
        int r = kill(pid, SIGTERM);
        if (r == -1)
            printf("engine", YLOG_WARN,
                   "kill %lld failed", (long long) pid);
        else
        {
            int status;
            waitpid(pid, &status, 0);
        }
    }
    if (profile_temp && profile_path)
    {
        struct stat buf;
        if (stat(profile_path, &buf) == 0)
        {
            WRBUF w = wrbuf_alloc();
            wrbuf_printf(w, "/bin/rm -r %s", profile_path);
            system(wrbuf_cstr(w));
            printf("engine", YLOG_LOG, "remove %s", profile_path);
            wrbuf_destroy(w);
        }
    }
    wrbuf_destroy(current_task);
    xfree(profile_path);
}

bool CF_Embed::wait_result_parent(const char **output, int timeout_ms, bool *sys_err)
{
    *sys_err = false;
    wrbuf_rewind(result_wrbuf);
    int round = 1;
    int step_ms = 10000; // 10 secs
    while (1)
    {
	char result_line[1024];
        int r = 0;
        struct pollfd fds;
        fds.fd = response_fd[0];
        fds.events = POLLIN;
        fds.revents = 0;

        r = poll(&fds, 1, timeout_ms > step_ms ? step_ms : timeout_ms);
        if (r == 0)
        {
            timeout_ms -= step_ms;
            if (timeout_ms > 0)
            {
                printf("engine", YLOG_LOG,
                               "poll waited for 10 secs. pid=%lld round=%d",
                               (long long) pid, round);
                round++;
                continue;
            }
            wrbuf_puts(result_wrbuf, "FAIL timeout");
            if (wrbuf_len(current_task) > 0)
            {
                wrbuf_puts(result_wrbuf, " - Task: ");
                wrbuf_puts(result_wrbuf, wrbuf_cstr(current_task));
            }
            else
                wrbuf_puts(result_wrbuf, " - no task");
            *sys_err = true;
            break;
        }
        else if (r == -1)
        {
            int x = errno;
            printf("engine", YLOG_LOG, "poll: %s", strerror(x));
            if (x == EINTR)
                continue;
            wrbuf_printf(result_wrbuf, "FAIL poll: %s", strerror(x));
            *sys_err = true;
            break;
        }
	r = read(response_fd[0], result_line, sizeof(result_line)-1);
        if (r == -1)
        {
            int x = errno;
            wrbuf_printf(result_wrbuf, "FAIL read: %s", strerror(x));
            *sys_err = true;
            break;
        }
	if (r < 1)
	    break;
	result_line[r] = '\0';
	char *nl = strchr(result_line, '\n');
	if (nl)
	{
	    *nl = '\0';
	    wrbuf_puts(result_wrbuf, result_line);
	    break;
	}
	wrbuf_puts(result_wrbuf, result_line);
    }
    const char *cp = wrbuf_cstr(result_wrbuf);
    if (output)
    {
	const char *cp1 = strchr(cp, ' ');
	*output = cp1 ? cp1+1 : "";
    }
    if (!memcmp(cp, "OK", 2))
    {
	return true;
    }
    return false;
}

void CF_Embed::get_result_parent(const char *default_msg, const char **output)
{
    bool sys_err = false;
    assert(output);
    int wait_task = 120;
    {
        char *str = getenv("CF_WAIT_TASK");
        if (str)
            wait_task = atoi(str);
    }
    if (!wait_result_parent(output, wait_task * 1000, &sys_err))
    {
        if (*output && **output)
        {
            if (sys_err)
            {
    	        printf("engine", YLOG_WARN, 
                       "sys err for %s", default_msg);
                throw CF_Engine_Error_System(*output);
            }
	    else
                throw CF_Engine_Error_Task(*output);
        }
        else
            throw CF_Engine_Error_Task(default_msg);
    }
}

void CF_Embed::load_cf_parent(const char *cf_file, const char *session_json)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("load_cf but no browser");
    }
    WRBUF w = wrbuf_alloc();
    if (session_json)
    {
      wrbuf_printf(w, "load_cf %s %s\n", cf_file, session_json);
    }
    else
    {
      wrbuf_printf(w, "load_cf %s\n", cf_file);
    }
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("load_cf", &output);
}

void CF_Engine::load_cf(const char *cf_file, const char *session_json)
{
    m_priv->load_cf_parent(cf_file, session_json);
}

void CF_Embed::save_cf_parent(const char *cf_file)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("save_cf but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "save_cf %s\n", cf_file);

    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("save_cf", &output);
}

void CF_Engine::save_cf(const char *cf_file)
{
    m_priv->save_cf_parent(cf_file);
}

void CF_Embed::run_script_parent(const char *script_file)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_script but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "run_script %s\n", script_file);

    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("run_script", &output);
}

void CF_Engine::run_script(const char *script_file)
{
    m_priv->run_script_parent(script_file);
}

/* function to ensure our input parameters are one line only (JSON) */
static void add_task_input(WRBUF w, const char *input)
{
    for (; *input; input++)
        if (strchr("\r\n", *input))
            wrbuf_putc(w, ' ');
        else
            wrbuf_putc(w, *input);
}

void CF_Embed::run_task_parent(const char *taskName, bool optional,
                               const char *input,
                               const char **output)
{
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_task but no browser");
    }
    wrbuf_rewind(current_task);
    wrbuf_puts(current_task, taskName);
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "%s %s ", optional ? "run_task_opt" : "run_task", taskName);
    add_task_input(w, input);
    wrbuf_puts(w, "\n");
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("run_task", output);
}

void CF_Engine::run_task(const char *taskName, const char *input,
			 const char **output)
{
    m_priv->run_task_parent(taskName, false, input, output);
}

void CF_Engine::run_task_opt(const char *taskName, const char *input,
                             const char **output)
{
    m_priv->run_task_parent(taskName, true, input, output);
}

bool CF_Embed::run_tests_parent(const char *task_order)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_task but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_puts(w, "run_tests ");
    add_task_input(w, task_order);
    wrbuf_puts(w, "\n");
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("run_tests", &output);
    return true;
}

bool CF_Engine::run_tests(const char *task_order)
{
    return m_priv->run_tests_parent(task_order);
}

bool CF_Embed::unit_test_parent(const char *name)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_task but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_puts(w, "unit_test - "); // task name is '-'
    add_task_input(w, name);
    wrbuf_puts(w, "\n");
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("unit_test", &output);
    return true;
}

bool CF_Engine::unit_test(const char *name)
{
    return m_priv->unit_test_parent(name);
}

bool CF_Embed::screen_shot_parent(const char *filename)
{
    const char *display_name = display.get_display();
    WRBUF w = wrbuf_alloc();

    wrbuf_printf(w, "xwd -id %lld ", (long long) main_window_xid);
    if (display_name)
        wrbuf_printf(w, "-display %s ", display_name);

    wrbuf_printf(w, "| xwdtopnm | pnmtopng >%s", filename);
    int ret = system(wrbuf_cstr(w));

    wrbuf_destroy(w);
    if (ret)
        return false;
    return true;
}

bool CF_Engine::screen_shot(const char *filename)
{
    return m_priv->screen_shot_parent(filename);
}

void CF_Embed::dom_string_parent(char **retval)
{
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "dom_string\n");
    write_request(w);
    wrbuf_destroy(w);
    get_result_parent("dom_string", (const char **) retval);
}

void CF_Engine::dom_string(char **retval)
{
    m_priv->dom_string_parent(retval);
}

CF_Engine_Error::CF_Engine_Error(const char *msg)
{
    this->msg = xstrdup(msg);
}

CF_Engine_Error::~CF_Engine_Error()
{
    xfree(msg);
}

const char *CF_Engine_Error::what()
{
    return msg;
}

const char *CF_Logger::logstr(int level)
{
    if (level & YLOG_FATAL)
        return "fatal";
    if (level & YLOG_DEBUG)
        return "debug";
    if (level & YLOG_WARN)
        return "warn";
    if (level & YLOG_LOG)
        return "log";
    return "-";
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

