/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file cfrun.cpp
 * \brief CF_Engine runner and shell
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <yaz/yaz-version.h>
#include <yaz/options.h>
#include <yaz/yaz-util.h>
#include <yaz/wrbuf.h>
#include "cf_factory.h"

static void usage()
{
    printf("cfrun [-c cfname] [-l loglevel] [-p proxy ] [-t] [-u name] [-V] [task args ...] \n");
    exit(1);
}

int main(int argc, char **argv)
{
    const char *task_name = 0;
    WRBUF task_args = wrbuf_alloc();
    const char *cf_file = 0;
    const char *cf_proxy = 0;
    const char *unit_name = 0;
    const char *cf_loglevel = 0;
    int ret;
    int exit_code = 0;
    char *arg;
    CF_Factory cf_factory(0);
    CF_Engine::Flags flags = CF_Engine::FORK;

    while ((ret = options("c:l:p:tu:V", argv, argc, &arg)) != -2)
    {
	switch(ret)
	{
	case 'c':
	    cf_file = arg;
	    break;
        case 'l':
            cf_loglevel = arg;
            break;
	case 'p':
	    cf_proxy = arg;
	    break;
	case 't':
            flags = CF_Engine::THREAD;
            break;
        case 'u':
            unit_name = arg;
            break;
        case 'V':
            puts(CF_VERSION);
            exit(1);
            break;
	case 0:
	    if (!task_name)
		task_name = arg;
	    else
	    {
		wrbuf_puts(task_args, arg);
		wrbuf_puts(task_args, " ");
	    }
	    break;
	default:
	    usage();
	}
    }
    CF_Engine *e = 0;
    try
    {
        e = cf_factory.create(flags, cf_proxy, cf_loglevel, 0, -1);

	if (cf_file)
	    e->load_cf(cf_file);

	if (task_name)
	{
	    const char *result = 0;
	    e->run_task(task_name, wrbuf_cstr(task_args), &result);
	    if (result)
		printf("Got result %s\n", result);
	}
	else if (unit_name)
        {
	    bool b = e->unit_test(unit_name);
            printf("%s\n", b ? "OK" : "FAIL");
            if (!b)
                exit_code = 1;
        }
        else
	{
	    char cmdline[1024];

	    while (1)
	    {
		if (isatty(0))
                {
		    printf("CF>"); fflush(stdout);
		}
		if (!fgets(cmdline, sizeof(cmdline)-1, stdin))
		    break;
		if (strlen(cmdline) > 0 && cmdline[strlen(cmdline)-1] == '\n')
		    cmdline[strlen(cmdline)-1] = '\0';

                char cmd[100];
                int no_chars = 0;

                const char *cmd_ptr = cmdline;
                if (sscanf(cmd_ptr, "%99s%n", cmd, &no_chars) < 1)
                    continue;

                cmd_ptr += no_chars;
                while (*cmd_ptr == ' ')
                    cmd_ptr++;

		if (!yaz_matchstr(cmd, "task"))
		{
                    char task_name[100];
                    if (sscanf(cmd_ptr, "%99s%n", task_name, &no_chars) > 0)
                    {
                        cmd_ptr += no_chars;
                        while (*cmd_ptr == ' ')
                            cmd_ptr++;
                        const char *result = 0;
                        e->run_task(task_name, cmd_ptr, &result);
                        if (result)
                            printf("%s\n", result);
                    }
                }
                else if (!yaz_matchstr(cmd, "task_opt"))
                {
                    char task_name[100];
                    if (sscanf(cmd_ptr, "%99s%n", task_name, &no_chars) > 0)
                    {
                        cmd_ptr += no_chars;
                        while (*cmd_ptr == ' ')
                            cmd_ptr++;

                        const char *result = 0;
                        e->run_task_opt(task_name, cmd_ptr, &result);
                        if (result)
                            printf("%s\n", result);
                    }
		}
		else if (!yaz_matchstr(cmd, "test"))
		{
                    bool ok = e->run_tests(cmd_ptr);
                    if (ok)
                        printf("OK\n");
                    else
                        printf("FAILED\n");
		}
		else if (!yaz_matchstr(cmd, "unittest"))
		{
                    bool b = e->unit_test(cmd_ptr);
                    printf("%s\n", b ? "OK" : "FAIL");
		}
		else if (!yaz_matchstr(cmd, "sleep"))
		{
		    if (*cmd_ptr)
			sleep(atoi(cmd_ptr));
		    else
			sleep(2);
		}
		else if (!yaz_matchstr(cmd, "cf"))
		{
		    e->load_cf(cmd_ptr);
		}
		else if (!yaz_matchstr(cmd, "write"))
		{
		    e->save_cf(cmd_ptr);
		}
                else if (!yaz_matchstr(cmd, "script"))
		{
		    e->run_script(cmd_ptr);
		}
                else if (!yaz_matchstr(cmd, "screen_shot"))
		{
                    e->screen_shot(cmd_ptr);
		}
                else if (!yaz_matchstr(cmd, "dom_string"))
		{
                    char *lalala = 0;
                    e->dom_string(&lalala);
		}
		else if (!yaz_matchstr(cmd, "quit"))
		{
		    break;
		}
		else if (!yaz_matchstr(cmd, "help"))
		{
		    printf("cf file                     - loads CF file\n");
		    printf("write file                  - saves CF file\n");
		    printf("script file                 - run Javascript from file in engine context\n");
		    printf("quit                        - quits program\n");
		    printf("task name {\"arg1\":\"value\"}  - runs task\n");
		    printf("task_opt name args          - runs task optional\n");
		    printf("test t1,t2,t3               - runs test of tasks given (in order)\n");
		    printf("unittest name               - runs unit name\n");
		    printf("sleep N                     - sleep N seconds\n");
		    printf("screen_shot fname           - to filename\n");
		}
		else if (*cmd)
		{
		    printf("Unknown command: %s\nType help for help\n", cmd);
		}
	    }
	}
    }
    catch (CF_Engine_Error &e) {
	fprintf(stderr, "CF Error: %s\n", e.what());
        exit_code = 2;
    }
    catch (...) {
	fprintf(stderr, "CF Error: Unknown Exception\n");
        exit_code = 3;
    }
    delete e;

    wrbuf_destroy(task_args);
    exit(exit_code);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

