/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief JSON parser program
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <yaz/xmalloc.h>
#include <yaz/options.h>

static void usage(void)
{
    printf("json_parse [-x][-f]\n");
    exit(1);
}
#include "json.h"
/** \brief main as usual */
int main(int argc, char **argv)
{
    char *arg = 0;
    int r;
    bool xml_parse = false;
    bool facet_parse = false;
    while ((r=options("xf", argv, argc, &arg)) != YAZ_OPTIONS_EOF)
    {
        switch(r)
        {
        case YAZ_OPTIONS_ERROR:
            fprintf(stderr, "Bad option: %s\n", arg);
            usage();
            exit(1);
        case 'x':
            xml_parse = true;
            break;
        case 'f':
            facet_parse = true;
            break;
        }
    }
    WRBUF w = wrbuf_alloc();
    int c;
    FILE *f = stdin;
    while ((c = getc(f)) != EOF)
        wrbuf_putc(w, c);
    const char *json_str = wrbuf_cstr(w);
    const char *err_msg;

    struct json_node *n = json_parse(json_str, &err_msg);
    if (!n)
    {
        fprintf(stderr, "JSON parse error: %s\n", err_msg);
        json_remove_node(n);
        exit(1);
    }
    wrbuf_rewind(w);
    if (facet_parse)
    {
        ODR odr = odr_createmem(ODR_ENCODE);
        ODR print  = odr_createmem(ODR_PRINT);
        Z_FacetList *facet_list = json_to_facet(n, odr);
        if (facet_list)
        {
            z_FacetList(print, &facet_list, 0, 0);
            wrbuf_puts(w, "{\"facets\":");
            facet_to_json(facet_list, w);
            wrbuf_puts(w, "}");
            puts(wrbuf_cstr(w));
        }
        odr_destroy(odr);
        odr_destroy(print);
    }
    else if (xml_parse)
    {
        json_to_xml(n, w, "opt");
        puts(wrbuf_cstr(w));
    }
    else
    {
        json_write_wrbuf(n, w);
        puts(wrbuf_cstr(w));
    }
    json_remove_node(n);
    wrbuf_destroy(w);
    exit(0);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
