/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief JSON to XML conversion
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <yaz/xmalloc.h>
#include <yaz/yaz-iconv.h>

#include "json.h"

static void open_tag(WRBUF wrb, const char *name)
{
    wrbuf_puts(wrb, "<");
    wrbuf_puts(wrb, name);
    wrbuf_puts(wrb, ">");
}

static void close_tag(WRBUF wrb, const char *name)
{
    wrbuf_puts(wrb, "</");
    wrbuf_puts(wrb, name);
    wrbuf_puts(wrb, ">\n");
}

static void make_xml(struct json_node *n, WRBUF wrb, struct json_node *parent,
                     int indent)
{
    if (!n)
        return;
    if (parent && n->type != json_node_list && n->type != json_node_array)
    {
        int i;
        if (wrbuf_len(wrb) > 0 && wrbuf_buf(wrb)[wrbuf_len(wrb)-1] == '>')
            wrbuf_putc(wrb, '\n');
        for (i = 0; i < indent; i++)
            wrbuf_putc(wrb, ' ');
        open_tag(wrb, parent->u.string);
    }
    switch (n->type)
    {
    case json_node_object:
        make_xml(n->u.link[0], wrb, 0, indent);
        break;
    case json_node_array:
        make_xml(n->u.link[0], wrb, parent, indent);
        break;
    case json_node_list:
        for (; n; n = n->u.link[1])
        {
            make_xml(n->u.link[0], wrb, parent, indent);
        }
        break;
    case json_node_pair:
        make_xml(n->u.link[1], wrb, n->u.link[0], indent + 2);
        break;
    case json_node_string:
        wrbuf_xmlputs(wrb, n->u.string);
        break;
    case json_node_number:
        wrbuf_printf(wrb, "%lg", n->u.number);
        break;
    case json_node_true:
        wrbuf_puts(wrb, "true");
        break;
    case json_node_false:
        wrbuf_puts(wrb, "false");
        break;
    case json_node_null:
        wrbuf_puts(wrb, "null");
        break;
    }
    if (n && parent && n->type != json_node_list && n->type != json_node_array)
    {
        if (n->type == json_node_object)
        {
            int i;
            for (i = 0; i < indent; i++)
                wrbuf_putc(wrb, ' ');
        }
        close_tag(wrb, parent->u.string);
    }
}

void json_to_xml(struct json_node *n, WRBUF wrb, const char *root_elem)
{
    assert(root_elem);
    open_tag(wrb, root_elem);
    make_xml(n, wrb, 0, 0);
    close_tag(wrb, root_elem);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
