/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief RPN to CF queries program
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <yaz/pquery.h>
#include <yaz/xmalloc.h>
#include <yaz/options.h>
#include "rpn_to_json.h"
#include "type7_sort.h"
#include "cf_metadata.h"
#include <yaz/diagbib1.h>

static void usage()
{
    printf("rpn_to_json [-h] [-c flags] [-d subdatabase] [-s] [-f] pqf\n");
    printf("  -h for this helptext\n");
    printf("  -c capability flags\n");
    printf("  -d subdatabase\n");
    printf("  -s for simple query (flattened to single fields) \n");
    printf("  -f for the fullquery (json tree of the full query)\n");
    printf("     if none given, defaults to both -s and -f\n");
}

int main(int argc, char **argv)
{
    const char *sub_database = 0;
    const char *capability_flags = 0;
    CF_Metadata md;
    int do_simple = 0;
    int do_full = 0;
    char *arg = 0;
    int r;
    WRBUF rpn_w = wrbuf_alloc();
    NMEM nmem = nmem_create();
    while ((r=options("c:sfhd:", argv, argc, &arg)) != YAZ_OPTIONS_EOF)
    {
        switch(r)
        {
        case YAZ_OPTIONS_ERROR:
            fprintf(stderr, "Bad option: %s\n", arg);
            usage();
            exit(1);
        case 'c':
            if (strstr(arg, ".cf"))
            {
                if (!md.parse_cf(arg))
                {
                    fprintf(stderr, "Missing or bad CF file: %s\n", arg);
                    exit(1);
                }
                capability_flags = md.get("flags");
            }
            else
                capability_flags = arg;
            break;
        case 'h':
            usage();
            exit(1);
        case 's':
            do_simple = 1;
            break;
        case 'f':
            do_full = 1;
            break;
        case 'd':
            sub_database = arg;
            break;
        case 0:
            if (wrbuf_len(rpn_w))
                wrbuf_puts(rpn_w, " ");
            wrbuf_puts(rpn_w, arg);
        }
    }
    if ( ! do_simple && ! do_full )
    {  // default to doing them both
        do_simple = 1;
        do_full = 1;
    }

    if (wrbuf_len(rpn_w) == 0)
    {
        fprintf(stderr, "Missing PQF query\n");
        usage();
        exit(1);
    }
    else
    {
        ODR o = odr_createmem(ODR_ENCODE);
        YAZ_PQF_Parser p = yaz_pqf_create();
        Z_RPNQuery *rpn = yaz_pqf_parse(p, o, wrbuf_cstr(rpn_w));
        if (rpn)
        {
            CF_RPN_to_JSON rpn_to_json;
            struct sort_elem *sort_list = 0;
            type7_sort(rpn, nmem, &sort_list);

            rpn_to_json.set_capability_flags(capability_flags);

            if ( do_full ) {
                if (rpn_to_json.parse_full(rpn, sub_database,sort_list, 0))
                {
                    const char *json_res;
                    if ((json_res = rpn_to_json.get_full()))
                    {
                        printf("%s\n", json_res);
                    }
                }
                else
                {
                    const char *addinfo;
                    int err = rpn_to_json.get_diagnostic(&addinfo);
                    fprintf(stderr, "RPN_TO_JSON full parse failed\n");

                    fprintf(stderr, "Diagnostic %d: %s\n", err,
                            yaz_diag_bib1_str(err));
                            if (addinfo)
                                fprintf(stderr, "Addinfo: %s\n", addinfo);
                            exit(1);
                }
            }

            if ( do_simple ) {
                if (rpn_to_json.parse(rpn, sub_database,sort_list))
                {
                    int i = 0;
                    const char *json_res;
                    while ((json_res = rpn_to_json.get_result(i)))
                    {
                        printf("%s\n", json_res);
                        i++;
                    }
                }
                else
                {
                    const char *addinfo;
                    int err = rpn_to_json.get_diagnostic(&addinfo);
                    fprintf(stderr, "RPN_TO_JSON parse failed\n");

                    fprintf(stderr, "Diagnostic %d: %s\n", err,
                            yaz_diag_bib1_str(err));
                    if (addinfo)
                        fprintf(stderr, "Addinfo: %s\n", addinfo);
                    exit(1);
                }
            }

        }
        else
        {
            const char *msg = 0;
            size_t off = 0;
            yaz_pqf_error(p, &msg, &off);
            fprintf(stderr, "Bad RPN: %s\n", msg);
            exit(1);
        }
        odr_destroy(o);
    }
    wrbuf_destroy(rpn_w);
    nmem_destroy(nmem);
    exit(0);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
/** \brief main as usual */
