/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief type-7 sort header
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>

#include <yaz/log.h>
#include <yaz/yaz-util.h>
#include <yaz/snprintf.h>
#include <yaz/wrbuf.h>

#include "type7_sort.h"

struct sort_element {
    char *field;
    char *flags;
    int prio;
    struct sort_element *next;
};

typedef struct {
    int type;
    int major;
    int minor;
    Z_AttributeElement **attributeList;
    int num_attributes;
} AttrType;

static void attr_init_APT(AttrType *src, Z_AttributesPlusTerm *zapt, int type)
{
    src->attributeList = zapt->attributes->attributes;
    src->num_attributes = zapt->attributes->num_attributes;
    src->type = type;
    src->major = 0;
    src->minor = 0;
}

static int attr_find_ex(AttrType *src, const Odr_oid **attribute_set_oid,
		 const char **string_value)
{
    int num_attributes;

    num_attributes = src->num_attributes;
    while (src->major < num_attributes)
    {
        Z_AttributeElement *element;

        element = src->attributeList[src->major];
        if (src->type == *element->attributeType)
        {
            switch (element->which)
            {
            case Z_AttributeValue_numeric:
                ++(src->major);
                if (element->attributeSet && attribute_set_oid)
                    *attribute_set_oid = element->attributeSet;
                return *element->value.numeric;
                break;
            case Z_AttributeValue_complex:
                if (src->minor >= element->value.complex->num_list)
                    break;
                if (element->attributeSet && attribute_set_oid)
                    *attribute_set_oid = element->attributeSet;
                if (element->value.complex->list[src->minor]->which ==
                    Z_StringOrNumeric_numeric)
                {
                    ++(src->minor);
                    return
                        *element->value.complex->list[src->minor-1]->u.numeric;
                }
                else if (element->value.complex->list[src->minor]->which ==
                         Z_StringOrNumeric_string)
                {
                    if (!string_value)
                        break;
                    ++(src->minor);
                    *string_value =
                        element->value.complex->list[src->minor-1]->u.string;
                    return -2;
                }
                else
                    break;
            default:
                assert(0);
            }
        }
        ++(src->major);
    }
    return -1;
}

static int attr_find(AttrType *src, const Odr_oid **attribute_set_id)
{
    return attr_find_ex(src, attribute_set_id, 0);
}

static void recur(Z_RPNStructure **s, NMEM nmem, struct sort_elem **sort_list,
                  int enable_type_7)
{
    int w = (*s)->which;
    if (w == Z_RPNStructure_simple)
    {
        Z_Operand *op = (*s)->u.simple;
        if (op->which == Z_Operand_APT && enable_type_7)
        {
            AttrType type_7;
            int type7_value;
            Z_AttributesPlusTerm *apt = op->u.attributesPlusTerm;

            attr_init_APT(&type_7, apt, 7);
            type7_value = attr_find(&type_7, 0);
            if (type7_value != -1)
            {
                AttrType use;
                const char *use_string = 0;
                int use_value = 0;
                int prio = 0;
                char flags[4];

                flags[0] = '\0';
                flags[1] = '\0';

                if (type7_value == 2)
                    flags[0] = 'r';

                attr_init_APT(&use, apt, 1);
                use_value = attr_find_ex(&use, 0, &use_string);

                if (apt->term->which == Z_Term_general)
                {
                    Odr_oct *oct = apt->term->u.general;
                    if (oct->len > 0 && oct->buf[0] >= '0' && oct->buf[0] <= '9')
                    {
                        prio = oct->buf[0] - '0';
                    }
                }

                struct sort_elem **sort_p = sort_list;
                while (*sort_p && prio > (*sort_p)->prio)
                    sort_p = &(*sort_p)->next;

                struct sort_elem *n_sort;
                n_sort = (struct sort_elem *)
                    nmem_malloc(nmem, sizeof(*n_sort));
                n_sort->next = *sort_p;
                *sort_p = n_sort;

                n_sort->prio = prio;
                if (use_value < 0)
                {
                    n_sort->field = nmem_strdup(nmem, use_string ?
                                                use_string : "");
                }
                else
                {
                    char use_str[40];
                    sprintf(use_str, "%d", use_value);
                    n_sort->field = nmem_strdup(nmem, use_str);
                }
                n_sort->flags = nmem_strdup(nmem, flags);
                /* got sort */
                *s = 0;
                return;
            }
        }
    }
    else if (w == Z_RPNStructure_complex)
    {
        Z_Complex *comp = (*s)->u.complex;
        recur(&comp->s1, nmem, sort_list, 1);
        recur(&comp->s2, nmem, sort_list, 1);
        if (!comp->s1)
            *s = comp->s2;
        else if (!comp->s2)
            *s = comp->s1;
    }
}


void type7_sort(Z_RPNQuery *q, NMEM nmem, struct sort_elem **sort_list)
{
    recur(&q->RPNStructure, nmem, sort_list, 0);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

