# Copyright (c) 2010 IndexData ApS. http://www.indexdata.com
#
# Facets.pm - create faceted classification lists

package Masterkey::Admin::Facets;

use strict;
use warnings;

use Encode;
use URI;
use HTML::Entities;
use Carp;

our $VERSION = '0.1';

sub new {
    my $class = shift;
    my $self  = {
        uri                 => "",
        formInputName       => "query",
        sort_by             => "hits",      # hits | alphabet
        max                 => 10,
        div_id             => "",          # <div id="...">
        div_class          => "facets",    # <div class="...">
        ignore_empty_facets => 1,
        cql_field           => "",

        words => {},
        @_
    };
    bless $self, $class;
    return $self;
}

#
# create a hash of new facets objects by facet name
# $facet = Masterkey::Admin::Facets->new_group( "facets" => ["category", "serviceprovider"], "uri"=>"http://indexdata.com" )
# $facet->{"serviceprovider"}->add("...");
#
sub new_group {
    my $class = shift;

    my %args   = @_;
    my $facets = $args{'facets'};
    delete $args{'facets'};

    # unique value per facet
    delete $args{'div_id'};

    die "No facets names configured!\n" if ref $facets ne 'ARRAY';

    my %objs;
    foreach my $f (@$facets) {
        $objs{$f} = new( $class, %args );
    }

    return \%objs;
}


sub group_from_xml {
    my $class = shift();
    my %args = @_;
    my $xml = delete $args{xml};

    my $group = $class->new_group(%args, facets => [ map { $_->{name} } @$xml ]);
    foreach my $field (@$xml) {
	foreach my $term (@{ $field->{term} }) {
	    $group->{$field->{name}}->add($term->{content}, $term->{count});
	}
    }

    return $group;
}


sub add {
    my $self  = shift;
    my($words, $incr) = @_;
    $incr = 1 if !defined $incr;

    return
      if $self->{'ignore_empty_facets'} && ( !defined $words || $words eq '' );

    $self->{'words'}->{$words} += $incr;
}

sub encode_CQL {
    my $term = shift;

    $term =~ s/"/\\"/g;
    return qq{"$term"};
}

# a wrapper for javascript save html: var js = html_js(...);
sub html_js {
    my $self = shift;

    my $js = $self->html(@_);
    $js =~ s/'/\\'/g;
    $js =~ s/\n/ /g;

    return $js;
}

sub html {
    my $self = shift;
    my %args = @_;

    my $uri =
        exists $args{'uri'} ? $args{'uri'}
      : defined $self->{'uri'} ? $self->{'uri'}
      :                       die "No uri defined!\n";
    my $formInputName =
        exists $args{'formInputName'} ? $args{'formInputName'}
      : $self->{'formInputName'}      ? $self->{'formInputName'}
      :                         die "No formInputName param name defined!\n";

    my $div_id =
        exists $args{'div_id'} ? $args{'div_id'}
      : $self->{'div_id'}      ? $self->{'div_id'}
      :                           "";
    my $div_class =
        exists $args{'div_class'} ? $args{'div_class'}
      : $self->{'div_class'}      ? $self->{'div_class'}
      :                              "";
    my $query = $args{'query'} || $self->{'query'};

    my $mapFn = $args{map_function} || $self->{map_function};

    ### It is PROFOUNDLY wrong that the so-called cql_field parameter
    #   also includes the relation by which the value is to be related
    #   to the field -- for example, "categories=" or "opacVendor==".
    my $cql_field_and_relation =
        exists $args{'cql_field'} ? $args{'cql_field'}
      : $self->{'cql_field'}      ? $self->{'cql_field'}
      :                             "";

    my $max =
      exists $args{'max'} ? $args{'max'} : $self->{'max'} ? $self->{'max'} : 10;
    $max = 10 if $max < 1 || $max > 100;

    my @words = ();
    my $words = $self->{'words'};

    if ( $self->{'sort_by'} eq 'hits' ) {
        @words = reverse sort { $words->{$a} <=> $words->{$b} } keys %$words;
    }
    elsif ( $self->{'sort_by'} ne 'alphabet' ) {
        @words = reverse sort keys %$words;
    }
    else {
        die "Unknown sort option: '", $self->{'sort_by'}, "'\n";
    }

    my $facets =
        qq{<div}
      . ( $div_id    ? qq{ id="$div_id"}       : "" )
      . ( $div_class ? qq{ class="$div_class"} : "" ) . ">\n";

    my $facets_counter = 0;
    foreach my $word (@words) {
        last if $facets_counter++ >= $max;

        my $url     = URI->new($uri);
        my $counter = $self->{'words'}{$word};
	my $fullquery = $cql_field_and_relation . encode_CQL($word);
	$fullquery = "$query and $fullquery" if defined $query;
        $url->query_form($formInputName, $fullquery);

	$word = &$mapFn($word) if $mapFn;
	if (defined $word) {
	    $facets .= '<a href="' . $url->as_string . '">' . encode_entities($word) . " ($counter)</a><br/>\n";
	}
    }

    $facets .= "</div>\n";

    return $facets;
}

1;

__DATA__

=head1 NAME

Facets - create faceted classification lists

=head1 SYNOPSIS

    use Masterkey::Admin::Facets;

    # a single facet object
    $facet_categories = Masterkey::Admin::Facets->new( 
	'uri' => "http://indexdata.com/",
	'div_id' => 'facet_categories', 
	'max' => 5, 
        'formInputName' => 'query', 
	'sort_by' => "hits",
        'div_class' => "facets",
        'ignore_empty_facets' => 1,
    );

    # a hash of objects for multiple facets
    $facet = Masterkey::Admin::Facets->new_group( 
	'facets' => ["categories", "serviceProvider"], 
	'max' => 5, 'uri' => "http://indexdata.com/");

    $facet->{"categories"}->add("foo");
    $facet->{"categories"}->add("bar");
    $facet->{"opacVendor"}->add("bla");

    % if ($showFacets) {
  	<p>Service Provider:</p> 
	<% $facet->{"serviceProvider"}->html('max'=>20, 'cql_field' => "serviceProvider=" ) %>
        <p>Categories:</p> 
	<% $facet->{"categories"}->html() %> 
    % }


=head1 FUNCTIONS

Available methods are:

=head2 new
	
	new(args...)

=cut

=head2 add

	add("term")

=cut

=head2 html

	html(args...)

=head2 html_js

	html_js(args...)

=cut


=head1 AUTHOR

Wolfram Schneider

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Facets


You can also look for information at:

=over 2

=item * Masterkey demo site

L<http://mk2.indexdata.com>

=back


=head1 ACKNOWLEDGEMENTS

Mike Taylor

=head1 COPYRIGHT & LICENSE

Copyright 2010 Index Data ApS, all rights reserved.

=cut

