package Masterkey::Admin::Utils;

use strict;
use warnings;
use Apache2::Request;
use Encode;
use Carp;

use Exporter 'import';
our @EXPORT_OK = qw(utf8param);

# Debugging wrapper: rename to utf8param() if needed
sub utf8param_wrapper {
    my($r, $key, $value) = @_;
    if (!defined $key) {
	my @res = _utf8param(@_);
	warn("utf8param(", join(", ", @_), ") ", @res ? ("-> '" . @res . "'") : "undefined");
	return @res;
    } else {
	my $res = _utf8param(@_);
	warn("utf8param(", join(", ", @_), ") ", $res ? ("-> '" . $res . "'") : "undefined");
	return $res;
    }
}

# Under Apache 2/mod_perl 2, the ubiquitous $r is no longer an
# Apache::Request object, nor even an Apache2::Request, but an
# Apache2::RequestReq ... which, astonishingly, doesn't have the
# param() method.  You make an Apache2::Request out of it, but even
# then you can't use it to SET parameter values, and it sometimes
# fails mysteriously, as in bug MKII-1475. For all these reasons,
# we're dumping it, and reverting to the boring old CGI module.

sub utf8param {
    my($q, $key, $value) = @_;

    confess("utf8param(): \$q undefined") if !defined $q;
    confess("utf8param(): \$q $q is not a CGI object") if ref $q ne "CGI";

    if (!defined $key) {
	# Request for all parameters in one go
	return map { decode_utf8($_) } $q->param();
    }

    my $raw = $q->param($key, $value);
    #warn("_utf8param[scalar]('$key') -> ", defined $raw ? "'$raw'" : "undef");

    return undef if !defined $raw;
    my $cooked = decode_utf8($raw);
    #warn "converted '$raw' to '", $cooked, "'\n" if $cooked ne $raw;
    return $cooked;
}


# Use this instead of Data::Dumper for more compact rendering

sub render_structure {
    my($item, $level) = @_;
    my $seenCount = 0;
    return _render_structure($item, $level, {}, \$seenCount);
}

sub _render_structure {
    use Scalar::Util qw(looks_like_number reftype isweak);

    my($item, $level, $seen, $seenCountRef) = @_;
    $level = 0 if !defined $level;

    return "undef" if !defined $item;

    #warn "item=$item, isweak=" . (isweak($item) ? "*TRUE*" : "false");
    if (isweak($item)) {
	return ref($item) . "(weak)";
    }

    my $which = $seen->{"$item"};
    return "VAR$which" if $which;
    $seen->{"$item"} = ++$$seenCountRef;

    my $reftype = ref $item;
    my $s = "";

  REALTYPE:
    if ($reftype eq "HASH") {
	$s .= "{";
	my @keys = sort keys %$item;
	for (my $i = 0; $i < @keys; $i++) {
	    $s .= "," if $i > 0;
	    my $key = $keys[$i];
	    $s .= " $key=" . _render_structure($item->{$key}, $level+1, $seen, $seenCountRef);
	}
	$s .= " " if @keys > 0;
	$s .= "}";
    } elsif ($reftype eq "ARRAY") {
	$s .= "[";
	for (my $i = 0; $i < @$item; $i++) {
	    $s .= "," if $i > 0;
	    $s .= " " . _render_structure($item->[$i], $level+1, $seen, $seenCountRef);
	}
	$s .= " " if @$item > 0;
	$s .= "]";
    } elsif ($reftype eq "CODE") {
	$s .= "CODE";
    } elsif ($reftype) {
	# An object
	$s = "$reftype=";
	$reftype = reftype($item);
	goto REALTYPE;
    } elsif (looks_like_number($item)) {
	$s = "$item";
    } else {
	# Non-numeric string
	my $t = $item;
	$t =~ s/\n/\\n/gs;
	if (length($t) > 30) {
	    $t = substr($t, 0, 30) . "...";
	}
	$s = '"' . $t . '"';
    }

    return $s;
}


if (0) {
    my $x = {}; bless $x, "Foo::Bar";
    my $y = [5,6]; bless $y, "Foo::Baz";
    my $z = [4, "ffoo", { x=>$x, z=>"mark", y=>$y }, 99 ];
    $x->{top} = $z;
    warn "isweak1=" . isweak($x->{top});
    Scalar::Util::weaken($x->{top});
    warn "isweak2=" . isweak($x->{top});
    print render_structure($z), "\n";
    use Data::Dumper;
    print Dumper($z);
}


1;
