package Masterkey::Admin::Torus;

use strict;
use warnings;
use Scalar::Util;
use LWP;
use HTML::Entities;
use HTTP::Request::Common; # for PUT
use URI::Escape qw(uri_escape_utf8);
use Compress::Zlib;
use XML::LibXML;
use Carp qw(confess);
use Masterkey::Admin::Record;
use Masterkey::Admin::ResultSet;
use Masterkey::Admin::Parent;
use Masterkey::Admin::Master;
use Masterkey::Admin::Properties;


our $using_XML_LibXML_Simple;

BEGIN {
    # Use XML::LibXML::Simple when possible, but it's not available on
    # all platforms (e.g. MacOS X v10.5.8) so be prepared to back down
    # to XML::Simple. Pathetically, the API of XML::LibXML::Simple is
    # not backwards compatible, so we need to know which we're
    # using. Sheesh.
    eval {
	require 'XML/LibXML/Simple.pm';
	import XML::LibXML::Simple;
	$using_XML_LibXML_Simple = 1;
    }; if ($@) {
	warn "can't load XML::LibXML::Simple -- falling back to XML::Simple\n";
	require 'XML/Simple.pm';
	import XML::Simple;
	$using_XML_LibXML_Simple = 0;
    }
}


sub new {
    my $class = shift();
    my($admin, $ws) = @_;

    die "WARNING: making $class with non-torus2 WS '$ws'"
	if $ws !~ /\/torus2\//;

    my $this = bless {
	admin => $admin,
	ws => $ws,
	ua => new LWP::UserAgent(),
	torusVer => undef,	# We'll discover this on first GET
    }, $class;

    if (my $httpAuth = $ENV{TORUS2AUTH}) {
	my($domain, $realm, $user, $pass) = split(/\//, $httpAuth, 4);
	$this->{ua}->credentials($domain, $realm, $user, $pass);
    }

    Scalar::Util::weaken($this->{admin});
    return $this;
}

sub admin { shift()->{admin} }
sub ws { shift()->{ws} }
sub torusVer { shift()->{torusVer} }


sub log {
    my $this = shift();
    my $admin = $this->admin();
    if (defined $admin) {
	return $admin->log(@_);
    } else {
	warn "weakened {admin} reference has become undefined: logging @_";
    }
}

sub _logTorusOp {
    my $this = shift();
    my($op, %args) = @_;

    my $ws = $this->ws();
    $ws =~ s/^.*?\/(torus2?|realm)\///;
    my $s = "";
    foreach my $name (sort keys %args) {
	my $val = $args{$name};
	next if $name eq 'listType' || !defined $val;
	$s .= ", " if $s ne "";
	$val = $val->field("identityId") if $name eq "user";
	$s .= "$name='$val'";
    }
    $this->log(Masterkey::Admin::LogLevel::TORUSOP, "$op($ws: $s)");
}


# Official entry points, including logging
sub records {
    my $this = shift();
    my($realm, $query, $user, $noConstraint, $facets, $start, $count, $recursive) = @_;

    return $this->_listByArgs(listType => "records",
			      realm => $realm,
			      query => $query,
			      user => $user,
			      noConstraint => $noConstraint,
			      facets => $facets,
			      start => $start,
			      count => $count,
			      recursive => $recursive);
}

sub world {
    my $this = shift();
    my($realm, $query, $facets, $start, $count, $recursive) = @_;

    return $this->_listByArgs(listType => "world",
			      realm => $realm,
			      query => $query,
			      facets => $facets,
			      start => $start,
			      count => $count,
			      recursive => $recursive);
}

sub merged {
    my $this = shift();
    my($realm, $query, $user, $noConstraint, $facets, $start, $count, $recursive) = @_;

    return $this->_listByArgs(listType => "merged",
			      realm => $realm,
			      query => $query,
			      user => $user,
			      noConstraint => $noConstraint,
			      facets => $facets,
			      start => $start,
			      count => $count,
			      recursive => $recursive);
}


# Internal entry point, includes logging
sub _listByArgs {
    my $this = shift();
    my(%args) = @_;

    my $query = $args{query};
    if (ref $query eq "HASH") {
	$args{query} = _hash2cql(%$query);
    }

    $this->_logTorusOp($args{listType}, %args);
    return $this->_listNologByArgs(%args);
}


# Internal entry point without logging
sub _listNologByArgs {
    my $this = shift();
    my(%args) = @_;

    my $uri = $this->_listURIbyArgs(%args);
    my $simple = $this->_getAsXML($uri);

    my $listType = $args{listType};
    my $ntotal = $simple->{total};
    my $nlocal = $simple->{local};
    my $noverrides = $simple->{overriden};
    my $nretrieved = $simple->{count};

    die "no count" if !defined $nretrieved;
    my $count = $args{count};
    die "impossible count $nretrieved>$count"
	if defined $count && $nretrieved > $count;

    my $start = $args{start};
    $start = 0 if !defined $start || $start eq '';
    my $start2 = $simple->{start};
    die "impossible start $start2 != $start"
	if defined $start2 && $start2 != $start;

    my @records;
    my $baseUri = $uri;
    $baseUri =~ s/\?.*//;
    foreach my $x (@{ $simple->{record} }) {
	my $isWorld;
	if ($listType eq 'world') {
	    $isWorld = 1;
	} elsif ($listType eq 'records') {
	    $isWorld = 0;
	} elsif ($listType eq 'merged') {
	    my $haveOverride = 0;
	    foreach my $layer (@{ $x->{layer} }) {
		$haveOverride = 1 if $layer->{name} eq 'override';
	    }
	    $isWorld = !$haveOverride;
	} else {
	    die "unexpected listType '$listType'";
	}

	my $rec = new Masterkey::Admin::Record($this, $isWorld, $x, $using_XML_LibXML_Simple);
	if (!$rec->uri()) {
	    # Newer Toruses don't set this, so fill it in by hand.
	    # This gluing process is pretty ugly.  ### Note, too, that
	    # it DOES NOT WORK for records in /merged/, which must
	    # instead be fetched from either /records/ or /world/
	    my $id = $rec->field("id");
	    die "id undefined" if !defined $id;
	    $rec->uri($baseUri . $id . "/");
	}
	push @records, $rec;
    }

    my $group;
    if (defined $args{facets}) {
	use Masterkey::Admin::Facets;
	$group = Masterkey::Admin::Facets->group_from_xml(
	    max => 5, uri => "", xml => $simple->{facets}->[0]->{facet});
    }

    return new Masterkey::Admin::ResultSet(\@records, $ntotal,
			(defined $nlocal ? $nlocal + $noverrides : undef),
			$nretrieved, $start, $group);
}


sub recordsURIbyArgs {
    my $this = shift();
    my(%args) = @_;

    my $realm = $args{realm};
    die "listURI invoked on 'world'" if defined $realm && $realm eq "world";
    return $this->_listURIbyArgs(listType => "records", %args);
}


# $user is only used if $query is defined and there is a Torus
# constraint, in which case it supplies values to be substituted into
# the constraint query.  That's it.  Otherwise, $user is unused.
#
sub _listURIbyArgs {
    my $this = shift();
    my(%args) = @_;

    if (0) {
	foreach my $key (sort keys %args) {
	    delete $args{$key} if !defined $args{$key};
	}
	warn("args = { " , join(", ", map { "$_='" . $args{$_} . "'" } sort keys %args) , " }");
    }

    my $listType = $args{listType};
    my $realm = $args{realm};
    my $layers = $args{layers};
    my $query = $args{query};
    my $user = $args{user};
    my $noConstraint = $args{noConstraint};
    my $facets = $args{facets};
    my $start = $args{start};
    my $count = $args{count};
    my $recursive = $args{recursive};

    $query = undef if defined $query && $query eq "";
    if (!$noConstraint && !$this->{isAuth} &&
	$this->admin()->conf("constrainIdentity")) {
	$query = $this->_addQuery($query, 'identityId=="${identityId}"', $user);
    }

    my $uri = $this->_torusURI($realm);
    $uri .= "$listType/";

    my %params = ();
    $params{layers} = $layers if defined $layers;
    $params{query} = $query if defined $query;
    $params{facets} = $facets if defined $facets;
    $params{start} = $start if defined $start;
    $params{count} = $count if defined $count;
    $params{recursive} = $recursive if defined $recursive;

    if (%params) {
	$uri .= "?" . join("&", map { uri_escape_utf8($_) . "=" .
				      uri_escape_utf8($params{$_}) }
			   sort keys %params);
    }

    #warn "_listURIbyArgs(listType=", _render($listType), " realm=", _render($realm), ", query=", _render($query), ", user=", _render($user), ", noConstraint=", _render($noConstraint), ") = '$uri'\n";
    return $uri;
}


sub _torusURI {
    my $this = shift();
    my($realm) = @_;

    # If $realm is undefined, then nothing is appended
    if (defined $realm && !$this->{isAuth}) {
	# Realm override only works on the primary Torus
	my $torusRealm = $this->admin()->conf("torusRealm");
	$realm = $torusRealm if defined $torusRealm;
    }

    my $uri = $this->ws();
    if (defined $realm) {
	$uri .= '.' if $uri !~ /\/$/;
	$uri .= $realm;
    }

    $uri .= '/';

    return $uri;
}


sub recordByArgs {
    my $this = shift();
    my(%args) = @_;

    $this->_logTorusOp("record", %args);
    return $this->_recordByArgs(listType => "records", %args);
}

sub worldRecordByArgs {
    my $this = shift();
    my(%args) = @_;

    $this->_logTorusOp("worldRecord", %args);
    return $this->_recordByArgs(listType => "world", %args);
}

sub _recordByArgs {
    my $this = shift();
    my(%args) = @_;

    my $url = $this->recordURIbyArgs(%args);
    my $simple = $this->_getAsXML($url);
    # Defend against subtle consequences of incorrect URIs.
    die "record() retrieved a list" if $simple->{record};
    die "record() retrieved a non-record" if !$simple->{layer};
    # Newer Toruses don't set this, so fill it in by hand
    $url =~ s/\?.*//;
    $simple->{URI} ||= $url;

    return new Masterkey::Admin::Record($this, $args{listType} eq 'world', $simple,
					$using_XML_LibXML_Simple);
}


sub recordURIbyArgs {
    my $this = shift();
    my(%args) = @_;

    # Safety check for an old "can't happen" that once happened
    my $realm = $args{realm};
    confess "recordURIbyArgs invoked on 'world'" if defined $realm && $realm eq "world";

    my $listURI = $this->_listURIbyArgs(%args, noConstraint => 1);

    ### Considing making _listURIbyArgs() do the following steps with $args{$id} if present
    my $id = $args{id};
    if ($listURI =~ s/\/\?/\/$id\/?/) {
	return $listURI;
    }
    return $listURI . "$id/";
}


sub _getAsXML {
    my $this = shift();
    my($uri) = @_;

    my $res = $this->_get($uri);
    my $xml = $res->content()
	or die "no content for '$uri'";

    return XMLin($xml, ForceArray => 1, KeyAttr => []);
}

sub _get {
    my $this = shift();
    my($uri) = @_;

    if ($uri =~ /^file:/) {
	# Hack needed for when we make fake Toruses in the filesystem,
	# since we can't have a file and a directory of the same name.
	$uri =~ s/\/$//;
	$uri .= ".xml";
    }

    $this->log(Masterkey::Admin::LogLevel::TORUS, "GET $uri");
    my $res = $this->{ua}->get($uri, "Accept-Encoding" => "gzip")
	or die "no response for '$uri'\n";
    $this->_decompress_content($res); ### Why doesn't LWP do this itself?
    if (!$res->is_success()) {
	$this->_http_fatal("GET", $uri, $res);
    }

    if (!defined $this->{torusVer}) {
	my $torusVer = $res->header("X-MK-Component");
	die "no X-MK-Component header: are you using mvn jetty:run instead of jetty:run-war?"
	    if !defined $torusVer;
	$torusVer =~ s/^Torus //
	    or die "torusVer '$torusVer' does not specify Torus";
	$torusVer =~ s/[^0-9.].*//;	# discard trailing "-SNAPSHOT" and similar
	$this->{torusVer} = $torusVer;
	#warn "set Torus's version to '$torusVer'";
    }

    return $res;
}

sub _decompress_content {
    my $this = shift();
    my($res) = @_;

    my $encoding = $res->header("content-encoding");
    if (defined $encoding && $encoding eq "gzip") {
	my $content = $res->content();
	my $oldlength = length($content);
	$content = Compress::Zlib::memGunzip($content);
	$res->content($content);
	$res->header("content-encoding" => "");
    	$this->log(Masterkey::Admin::LogLevel::COMPRESS,
		   "decompressed $oldlength to ", length($content), " chars ",
		   "(", int(100 * $oldlength / length($content)), "%)");
    } else {
    	$this->log(Masterkey::Admin::LogLevel::COMPRESS,
		   "read ", length($res->content()), " chars, not compressed");
    }
}


sub _put {
    my $this = shift();
    return $this->_put_or_post('PUT', Masterkey::Admin::LogLevel::WSPUT, \&PUT, @_);
}

sub _post {
    my $this = shift();
    return $this->_put_or_post('POST', Masterkey::Admin::LogLevel::WSPOST, \&POST, @_);
}

sub _put_or_post {
    my $this = shift();
    my($label, $loglevel, $method, $uri, $xml) = @_;

    $this->log(Masterkey::Admin::LogLevel::TORUS, "x$label $uri");
    $this->log($loglevel, $xml);
    my $res = $this->{ua}->request($method->($uri, Content => $xml,
		   "Content-Type" => "application/xml"))
	or die "no response for $label to '$uri'\n";

    if (!$res->is_success()) {
	$this->_http_fatal("$label to", $uri, $res);
    }

    return $res;
}


# $goneHash is a reference to a hash whose keys are the names of
# fields that should be removed completely from the record.
sub update {
    my $this = shift();
    my($master, $realm, $id, $type, $goneHash, %data) = @_;

    $this->_logTorusOp("update", master=>$master, realm=>$realm, id=>$id, type=>$type);
    my $uri;
    if ($master) {
	$uri = $this->masterURI($realm, $id);
    } else {
	$uri = $this->recordURIbyArgs(listType => "records", realm => $realm, id => $id);
    }
    my $xml = $this->_makeXML($master, $type, $goneHash, %data);
    $this->_put($uri, $xml);
}

sub add {
    my $this = shift();
    my($master, $realm, $worldId, $user, %data) = @_;
    my $admin = $this->admin();

    die "add invoked on 'world'" if defined $realm && $realm eq "world";
    $this->_logTorusOp("add", master=>$master, realm=>$realm, worldId=>$worldId, user=>$user);

    if (!$master) {
	$this->_check_unique($realm, $user, %data);
    }

    my %extra = ();
    if ($admin->conf("constrainIdentity")) {
	%extra = (identityId => $user->field("identityId"));
    }

    my @fields = $admin->fieldsWithDefaultValues();
    foreach my $field (@fields) {
	if (!defined $data{$field}) {
	    $extra{$field} = $admin->conf("field.$field.defaultValue");
	    warn "set default value $field='" . $extra{$field} . "'";
	}
    }

    $data{worldId} = $worldId if defined $worldId;
    my($uri, $xml);
    if ($master) {
	$uri = $this->mastersURI($realm);
    } else {
	$uri = $this->recordsURIbyArgs(realm => $realm, user => $user, noConstraint => 1);
    }
    $xml = $this->_makeXML($master, undef, {}, %data, %extra);
    my $res = $this->_post($uri, $xml);

    # Content is now empty, but the Location header has something like:
    #	http://coffee.indexdata.dk/torus/records/somelib/A1-0
    # from which we can extract the ID of the new record.
    die "unexpected content for POST to '$uri': " . $res->content()
	if $res->content();
    my $location =  $res->header('Location');
    my $id = $location;
    $id =~ s/\/$//; # Stupid Torus2 ends ALL URLs with a pointless slash
    $id =~ s/.*\///;

    return $id;
}

sub deleteByArgs {
    my $this = shift();
    my(%args) = @_;

    $this->_logTorusOp("delete", %args);
    my $uri = $this->recordURIbyArgs(listType => "records", %args);
    $this->_delete_by_uri($uri);
}

sub disableByArgs {
    my $this = shift();
    my(%args) = @_;

    $this->_logTorusOp("disable", %args);
    my $uri = $this->recordURIbyArgs(listType => "records", %args);
    my $xml = $this->_makeXML(0, undef, {}, disabled => "yes");
    $this->_put($uri, $xml);
}

sub enableByArgs {
    my $this = shift();
    my(%args) = @_;

    $this->_logTorusOp("enable", %args);
    my $uri = $this->recordURIbyArgs(listType => "records", %args);
    my $xml = $this->_makeXML(0, undef, {}, disabled => "no");
    $this->_put($uri, $xml);
}

sub _delete_by_uri {
    my $this = shift();
    my($uri) = @_;

    $this->log(Masterkey::Admin::LogLevel::TORUS, "DELETE $uri");

    # The HTTP::Request::Common module does not directly support the
    # DELETE method, for some reason.
    my $res = $this->{ua}->request(new HTTP::Request(DELETE => $uri))
	or die "no response for DELETE from '$uri'\n";

    if (!$res->is_success()) {
	$this->_http_fatal("DELETE from", $uri, $res);
    }
}

# Returns a list of all the values that the nominated field takes in
# the specified realm.  May optionally be limited by query, and will
# in any case be limited by any constrainIdentity specified in the tab
# configuration; $user is needed only to substitute query values into
# the constraint strings.
#
# One day, we'll probably have a Torus WS for scanning, but for now we
# just do it by hand: fetch and parse all the records, then throw them
# all away.
#
sub scan {
    my $this = shift();
    my($realm, $query, $user, $field) = @_;

    $this->_logTorusOp("scan", realm=>$realm, query=>$query,
		      user=>$user, field=>$field);

    my $rs = $this->_listNologByArgs(listType => "records", realm => $realm, query => $query, user => $user);
    my $records = $rs->records();
    my %register;
    foreach my $record (@$records) {
	my $val = $record->field($field);
	$register{$val}++ if defined $val;
    }

    return sort keys %register;
}


sub user1 {
    my $this = shift();

    if (@_ == 2 && $_[0] eq "id") {
	# Special-case a nicer query for the common lookup by ID
	return $this->recordByArgs(realm => undef, id => $_[1]);
    }

    my $rs = $this->records(undef, { @_ });
    my $users = $rs->records();
    return undef if @$users == 0;
    die "$this->user1(): " . scalar(@$users) . " hits" if @$users > 1;
    return $users->[0];
}


# Dies if a field in the profile that is marked unique in the
# configuration has a value the same as that field in an existing
# record.  Of course, this is weak -- there's a race condition.
#
sub _check_unique {
    my $this = shift();
    my($realm, $user, %data) = @_;

    my @conds;
    foreach my $ref ($this->admin()->configObject()->profile()->fields()) {
	my($name) = @$ref;
	if ($this->admin()->conf("field.$name.unique")) {
	    my $val = $data{$name};
	    push @conds, [ $name, $val ]
		if defined $val && $val ne "";
	}
    }

    return if !@conds;
    ### This is worryingly similar to _hash2cql()
    my $cql = join(" and ", map {
	my($name, $val) = @$_;
	$val =~ s/"/\\"/g;
	qq[$name == "$val"];
    } @conds);

    my $rs = $this->_listNologByArgs(listType => "records", realm => $realm, query => $cql, user => $user);
    my $records = $rs->records();
    return if !@$records;

    $cql =~ s/ == / /g; # A bit of a hack here
    die "There is already a record with $cql.\n";
}

sub _hash2cql {
    my(%args) = @_;

    my $query = join(" and ", map {
	my $val = $args{$_};
	$val =~ s/[""*?]/\\$&/g;
	qq[$_=="$val"];
    } sort keys %args);
    
    return $query;
}

sub _http_fatal {
    my $this = shift();
    my($method, $uri, $res) = @_;

    my $message = "can't $method '$uri': " . $res->message();
    my $content = $res->content();
    $message .= " ($content)" if defined $content && $content ne "";
    $message .= "\n";
    warn $message; # For the HTTP server log
    die $message;
}

sub _makeXML {
    my $this = shift();
    my($master, $type, $goneHash, %data) = @_;

    $type = $this->admin()->conf('type')
	if !defined $type;

    my %allKeys = (%data, %$goneHash);

    my $doc = XML::LibXML::Document->new('1.0', 'utf-8');
    my $root = $doc->createElement("record");
    $root->setAttribute(type => $type);

    my $layer = $doc->createElement('layer');
    $layer->setAttribute(name => $master ? 'master-override' : 'override');
    $root->appendChild($layer);

    for my $name (sort keys %allKeys) {
	if ($name =~ s/^@//) {
	    # Set attribute instead of field
	    $layer->setAttribute($name => $data{'@' . $name});
	    next;
	}

	my $tag = $doc->createElement($name);
	if (exists $goneHash->{$name}) {
	    $tag->setAttribute(gone => 'yes');
	} else {
	    $tag->appendTextNode($data{$name});
	}
	$layer->appendChild($tag);
    }

    $doc->setDocumentElement($root);
    return $doc->toString(1);
}

# You may pass either a simple hash or a Record object containing values
sub _substitute {
    my $this = shift();
    my($in, $obj) = @_;

    confess "_substitute() with no object" if !defined $obj;
    my $out = "";
    while ($in =~ s/(.*?)\${(.*?)}//) {
	my $val = ref($obj) eq "HASH" ? $obj->{$2} : $obj->field($2);
	$out .= $1 . $val;
    }

    return "$out$in";
}

sub _render {
    my($val) = @_;
    return "undef" if !defined $val;
    return "'$val'";
}

sub _addQuery {
    my $this = shift();
    my($query, $constraint, $user) = @_;

    my $clause = $this->_substitute($constraint, $user);
    if (!defined $query) {
	return $clause;
    }

    my $sort;
    # Hideous hack.  We should do better but don't have time.
    if ($query =~ /(.*) sortby (.*)/) {
	($query, $sort) = ($1, $2);
    }

    # Special case to avoid pointless ANDs
    if ($query =~ /cql\.allRecords\s*=\s*1/) {
	$query = $clause;
    } else {
	$query = "($query) and $clause";
    }

    $query .= " sortby $sort" if defined $sort;
    return $query;
}


sub make_realms_for_record {
    my $this = shift();
    my(%data) = @_;
    my $admin = $this->admin();

    # Add new realms as required
    my $i = 1;
    while (1) {
	my $realm = $admin->conf("realmOnNew.$i.url")
	    or last;
	my $type = $admin->conf("realmOnNew.$i.type")
	    or die "realmOnNew.$i.type not specified";

	my $uri = $this->_substitute($realm, \%data);
	$this->create_realm($uri, $type, 0);

	my $j = 1;
	while (1) {
	    my $pname = $admin->conf("realmOnNew.$i.parent.$j.name")
		or last;
	    my $purl = $admin->conf("realmOnNew.$i.parent.$j.url")
		or die "realmOnNew.$i.parent.$j.url not specified";
	    my $ppri = $admin->conf("realmOnNew.$i.parent.$j.priority")
		or die "realmOnNew.$i.parent.$j.priority not specified";
	    my $prefresh = $admin->conf("realmOnNew.$i.parent.$j.refreshAfter") || 0;

	    $this->_addParentByURI("${uri}parents/", $pname, $ppri, $prefresh, $purl);
	    $j++;
	}

	$i++;
    }
}


sub remove_realms_for_record {
    my $this = shift();
    my($identityId) = @_;

    my %data = (identityId => $identityId);
    my $admin = $this->admin();

    # Remove realms as required
    my $i = 1;
    while (1) {
	my $realm = $admin->conf("realmOnNew.$i.url")
	    or last;

	my $uri = $this->_substitute($realm, \%data);
	$this->_delete_realm($uri);
	$i++;
    }
}


sub create_realm {
    my $this = shift();
    my ($uri, $type, $autoInherit) = @_;

    my $xml = qq[<realm type="$type" name="this will be ignored"];
    $xml .= ' autoInherit="yes"' if $autoInherit;
    $xml .= '>';

    if ($type eq "searchable") {
	$xml .= qq[
  <matchKey>
    <field name="displayName" required="yes"/>
    <field name="serviceProvider" required="no"/>
  </matchKey>\n];
    }
    $xml .= qq[</realm>];
    $this->_logTorusOp("newRealm", realm=>$uri);
    $this->_put($uri, $xml);
}


sub _delete_realm {
    my $this = shift();
    my ($uri) = @_;

    $this->_logTorusOp("deleteRealm", realm=>$uri);
    $this->_delete_by_uri($uri);
}


sub _addParentByURI {
    my $this = shift();
    my($uri, $pname, $ppri, $prefresh, $purl) = @_;

    $this->_logTorusOp("addParent", uri=>$uri, name=>$pname,
		      priority=>$ppri, refreshAfter=>$prefresh, url=>$purl);
    my $xml = _parent_xml(undef, $pname, $ppri, $prefresh, $purl);
    my $res = $this->_post($uri, $xml);

    # See add() in ../Torus.pm for explanation of what follows
    die "unexpected content for POST to '$uri': " . $res->content()
	if $res->content();
    my $location =  $res->header('Location');
    my $id = $location;
    $id =~ s/\/$//; # Stupid Torus2 ends ALL URLs with a pointless slash
    $id =~ s/.*\///;

    return $id;
}

sub _addMasterByURI {
    my $this = shift();
    my($uri, $label, $matchQuery, $goneHash, %data) = @_;

    $this->_logTorusOp("addMaster", uri=>$uri, label=>$label, matchQuery=>$matchQuery);
    my $xml = _master_xml(undef, $label, $matchQuery, $goneHash, %data);
    my $res = $this->_post($uri, $xml);
}

sub parent {
    my $this = shift();
    my ($realm, $id) = @_;

    $this->_logTorusOp("parent", realm=>$realm, id=>$id);
    my $uri = $this->parentURI($realm, $id);
    my $simple = $this->_getAsXML($uri);
    return new Masterkey::Admin::Parent($this, $simple);
}


sub parents {
    my $this = shift();
    my ($realm, $query) = @_;

    $this->_logTorusOp("parents", realm=>$realm, query=>$query);
    my $uri = $this->parentsURI($realm, $query);
    my $simple = $this->_getAsXML($uri);

    return (map { new Masterkey::Admin::Parent($this, $_) }
	    @{ $simple->{parent} });
}

sub master {
    my $this = shift();
    my ($realm, $id) = @_;

    $this->_logTorusOp("master", realm=>$realm, id=>$id);
    my $uri = $this->masterURI($realm, $id);
    my $simple = $this->_getAsXML($uri);
    return new Masterkey::Admin::Master($this, $simple, $using_XML_LibXML_Simple);
}


sub masters {
    my $this = shift();
    my ($realm, $query) = @_;

    $this->_logTorusOp("masters", realm=>$realm, query=>$query);
    my $uri = $this->mastersURI($realm, $query);
    my $simple = $this->_getAsXML($uri);

    return (map { new Masterkey::Admin::Master($this, $_, $using_XML_LibXML_Simple) }
	    @{ $simple->{record} });
}

sub addParent {
    my $this = shift();
    my($realm, $name, $priority, $refreshAfter, $url) = @_;

    return $this->_addParentByURI($this->parentsURI($realm),
				  $name, $priority, $refreshAfter, $url);
}

sub addMaster {
    my $this = shift();
    my($realm, $label, $matchQuery, $goneHash, %data) = @_;

    return $this->_addMasterByURI($this->mastersURI($realm), $label, $matchQuery, $goneHash, %data);
}

sub updateParent {
    my $this = shift();
    my($realm, $id, $name, $priority, $refreshAfter, $url) = @_;

    $this->_logTorusOp("updateParent", realm=>$realm, id=>$id, name=>$name,
		      priority=>$priority, refreshAfter=>$refreshAfter, url=>$url);

    my $uri = $this->parentURI($realm, $id);
    my $xml = _parent_xml($id, $name, $priority, $refreshAfter, $url);
    $this->_put($uri, $xml);
}


sub _parent_xml {
    my($id, $name, $priority, $refreshAfter, $url) = @_;

    my $xml = '<parent ';
    $xml .= 'id="' . encode_entities($id) . '" ' if defined $id;
    $xml .= ('name="' . encode_entities($name) . '" ' .
	     'priority="' . encode_entities($priority) . '" ' .
	     'refreshAfter="' . encode_entities($refreshAfter) . '" ' .
	     'url="' . encode_entities($url) . '" ' .
	     '/>');
    return $xml;
}


sub deleteParent {
    my $this = shift();
    my($realm, $id) = @_;

    $this->_logTorusOp("deleteParent", realm=>$realm, id=>$id);
    my $uri = $this->parentURI($realm, $id);
    $this->_delete_by_uri($uri);
}


sub parentURI {
    my $this = shift();
    my ($realm, $id) = @_;

    my $uri = $this->parentsURI($realm);
    return "$uri$id/"; ### stupid trailing slash: Torus bug #4135
}


sub parentsURI {
    my $this = shift();
    my ($realm, $query) = @_;

    my $uri = $this->ws();
    $uri .= ".$realm" if defined $realm;
    $uri .= "/parents/";
    die "query not supported in parents()" if defined $query && $query ne "";
    return $uri;
}

### disturbingly similar to _makeXML()
sub _master_xml {
    my($id, $label, $matchQuery, $goneHash, %data) = @_;

    my %allKeys = (%data, %$goneHash);

    my $doc = XML::LibXML::Document->new('1.0', 'utf-8');
    my $root = $doc->createElement("record");
    $root->setAttribute(type => 'master');

    my $layer = $doc->createElement('layer');
    $layer->setAttribute(name => 'master-override');
    $layer->setAttribute(label => $label);
    $layer->setAttribute(matchQuery => $matchQuery);
    $root->appendChild($layer);

    for my $name (sort keys %allKeys) {
	my $tag = $doc->createElement($name);
	if (exists $goneHash->{$name}) {
	    $tag->setAttribute(gone => 'yes');
	} else {
	    $tag->appendTextNode($data{$name});
	}
	$layer->appendChild($tag);
    }

    $doc->setDocumentElement($root);
    return $doc->toString(1);
}


sub deleteMaster {
    my $this = shift();
    my($realm, $id) = @_;

    $this->_logTorusOp("deleteMaster", realm=>$realm, id=>$id);
    my $uri = $this->masterURI($realm, $id);
    $this->_delete_by_uri($uri);
}


sub masterURI {
    my $this = shift();
    my ($realm, $id) = @_;

    my $uri = $this->mastersURI($realm);
    return "$uri$id/";
}


sub mastersURI {
    my $this = shift();
    my ($realm, $query) = @_;

    my $uri = $this->ws();
    $uri .= ".$realm" if defined $realm;
    $uri .= "/masters/";
    die "query not supported in masters()" if defined $query && $query ne "";
    return $uri;
}

sub properties {
    my $this = shift();
    my ($realm) = @_;

    $this->_logTorusOp("properties", realm=>$realm);
    my $uri = $this->_torusURI($realm);
    my $simple = $this->_getAsXML($uri);
    return new Masterkey::Admin::Properties($this, $realm, $simple);
}


# Example XML to use here:
#
#	<?xml version="1.0" encoding="UTF-8"?>
#	<realm name="searchable.mike" type="searchable">
#	  <matchKey>
#	    <field name="displayName" required="yes"/>
#	    <field name="serviceProvider" required="no"/>
#	  </matchKey>
#	</realm>
#
sub setProperties {
    my $this = shift();
    my($realm, $name, $type, $matchKey, $autoInherit) = @_;

    $this->_logTorusOp("setProperties", realm=>$realm, name=>$name,
		       type=>$type, matchKey=>$matchKey,
		       autoInherit=>$autoInherit);

    my $ename = encode_entities($name);
    my $etype = encode_entities($type);
    my $maybeAI = defined $autoInherit ? ' autoInherit="yes"' : '';

    my $xml = qq[<?xml version="1.0" encoding="UTF-8"?>\n
<realm name="$ename" type="$etype"$maybeAI>\n];
    if ($matchKey) {
	$xml .= "  <matchKey>\n";
	# Grrr! Can't use XMLout as XML::LibXML::Simple doesn't have it!
	#$xml .= XMLout({ matchKey => $matchKey });
	foreach my $ref (@{ $matchKey->[0]->{field} }) {
	    my $name = encode_entities($ref->{name});
	    my $required = encode_entities($ref->{required});
	    $xml .= qq[    <field name="$name" required="$required"/>\n];
	}
	$xml .= "  </matchKey>\n";
    }
    $xml .= "</realm>\n";

    my $uri = $this->_torusURI($realm);
    $this->_post($uri, $xml);
}


sub sortedRecords {
    my $this = shift();
    return $this->_sortedList(0, @_);
}

sub sortedWorld {
    my $this = shift();
    my($realm, $query, $start, $count) = @_;
    return $this->_sortedList(1, $realm, $query, undef, $start, $count);
}

sub sortedMerged {
    my $this = shift();
    return $this->_sortedList(2, @_);
}

sub _sortedList {
    my $this = shift();
    my($listType, $realm, $query, $user, $start, $count) = @_;

    if (!defined $query || $query eq "") {
	$query = "cql.allRecords=1";
    }
    $query .= " sortby displayName/sort.ignoreCase";

    my @facetNames;
    # The odd "categories:\054" tells the Torus to split the categories field on commas.
    if ($this->admin()->conf("showFacets")) {
	my $facets = $this->admin()->conf("facetList");
	foreach my $facet (split(/\|/, $facets)) {
	    my $sep = $this->admin()->conf("facet.$facet.separator");
	    if (defined $sep) {
		$facet .= ':' . $sep;
	    }
	    push @facetNames, $facet;
	}
    }
    push @facetNames, "displayName"
	if $this->admin()->conf("showTagCloud");
    my $facetNames;
    $facetNames = join (",", @facetNames)
	if @facetNames;

    my $rs;
    if ($listType == 0) {
	$rs = $this->records($realm, $query, $user, undef, $facetNames, $start, $count);
    } elsif ($listType == 1) {
	$rs = $this->world($realm, $query, $facetNames, $start, $count);
    } elsif ($listType == 2) {
	$rs = $this->merged($realm, $query, $user, undef, $facetNames, $start, $count);
    } else {
	die "bad listType=$listType";
    }

    $rs->cloud($this->generateCloud($rs->facets()));
    return $rs;
}

sub generateCloud {
    my $this = shift();
    my($facets) = @_;

    my $admin = $this->admin();
    return undef
	if !$admin->conf("showTagCloud");

    require 'Masterkey/Admin/TagCloud.pm';
    my $words = $facets->{displayName}->{words};
    my $cloud = Masterkey::Admin::TagCloud->new();
    foreach my $displayName (keys %$words) {
	my $count = $words->{$displayName};
	foreach my $word (split(/[\W_-]+/, $displayName)) {
	    $cloud->add_term($word, $count) if length($word) > 1;
	}
    }

    $cloud->generate('uri' => ".", 'levels'=> $admin->conf("tagCloudLevels"));
    return $cloud;
}


1;
