package Masterkey::Admin::ServiceProxyStatistic;

use Time::HiRes qw/gettimeofday tv_interval/;
use Data::Dumper;
use File::Temp qw/ tempfile/;

use strict;
use warnings;

sub new {
    my $class = shift;

    my $self = {
        'debug'  => 0,
        'hostName' => "",
        'program' =>
          '/usr/share/masterkey/service-proxy-statistic/bin/sp-statistics',
        'logfile' =>
'/var/log/masterkey/service-proxy/quicksearch.ohiolink.edu-statistics.log',
        'program_opt' => ["--output-format=csv"],

        @_
    };

    bless $self, $class;

    $self->read_config;
    return $self;
}

sub debug { shift->{'debug'} }

sub read_config {
    my $self = shift;

    my $admin  = $self->{'admin'};
    my $hostName = $self->{'hostName'};

    return if !$admin;

    my $program = $admin->conf("serviceProxyStatisticProgram");
    my $logfile = $admin->conf("serviceProxyStatisticLogfile");

    # per user library log file?
    my $logfile_hostName = $admin->conf("serviceProxyStatisticHostNameLogfile");
    if ( $hostName && $logfile_hostName ) {
        $logfile = $self->get_path(
            'path'  => $logfile_hostName,
            'param' => { 'hostName' => $hostName }
        );
    }
    warn "logfile=$logfile\n" if $self->debug >= 1;

    $self->{"program"} = $program if $program;

    if ($logfile) {

        chomp($logfile);

        # space separated list of log files in mkadmin config
        if ( $logfile =~ /\s+/ ) {
            $logfile = [ split( /\s+/, $logfile ) ];
        }

        my @list = $self->expand_log_rotation(ref ($logfile) eq 'ARRAY' ? @$logfile : $logfile );
        $self->{"logfile"} = [ @list ];
    }
}

#
# expand log file rotation files
#
# foo.log -> foo.log.3 foo.log.2 foo.log.1 foo.log
#
# oldes logfiles will be first
#
sub expand_log_rotation {
    my $self = shift;
    my @logfiles = @_;

    my @list;
    my @files = map { ${_} . '*' } @logfiles;
    foreach my $file (@files) {
          push @list, `ls -tr $file`;
    }

    # remove newlines
    return map { chomp($_); $_ } @list;
}

sub get_path {
    my $self = shift;

    my %args = @_;

    my $path  = $args{'path'};
    my $param = $args{'param'};

    $path =~ s,%{(.*?)},$param->{$1},ge;

    warn "replace @{[ $args{'path'} ]} with $path\n" if $self->debug;
    return $path;
}

sub out {
    my $self = shift;

    my %args = @_;

    my $http_format = $args{'http_format'};
    my $r           = $args{'r'};
    my @opt         = @{ $args{'args'} };

    my $program     = $self->{'program'};
    my $logfile     = $self->{'logfile'};
    my @program_opt = @{ $self->{'program_opt'} };

    push @program_opt, @opt if scalar(@opt) > 0;

    my @logfiles = ref $logfile eq 'ARRAY' ? @{$logfile} : $logfile;

    my $t0 = [gettimeofday];

    my $tmp =
      File::Temp->new( TEMPLATE => '/tmp/sp-stat-XXXXXXXX', SUFFIX => '.stat' );
    my $tmpfile = $tmp->filename;

    # store output in a file. This avoid race condition with stdout on mod_perl
    push @program_opt, "--output-file=$tmpfile";

    if (grep { /^--storable-read=.*db$/} @opt) {
        warn "No need to use logfiles, use from storable instead\n" if $self->debug;
        undef @logfiles;
    }


    my @system = ( $program, @program_opt, @logfiles );
    my $message = "Execute: " . join( " ", @system );

    eval {
        if ( system(@system) )
        {
            warn "system @system failed: code: $?, message: $@\n";
            die "Error code: $?\n";
        }
    };

    # sent out a HTTP 503 error message as plain text
    if ($@) {
        $r->content_type("text/plain");
        $r->status(503);
        print
"Oops, an internal error occoured. Please contact the mkadmin administrator!\n";
        print "$@";
        return;
    }

    my $fh   = new IO::File $tmpfile;
    my $type = "";
    if ( $http_format =~ /^(xls|excel)$/ ) {
        $type = "application/vnd.ms-excel";
    }
    elsif ( $http_format =~ /^(xml)/ ) {
        $type = "application/$http_format";
    }
    elsif ( $http_format =~ /^(text|plain|sutrs)/ ) {
        $type = "text/plain";
    }
    elsif ( $http_format =~ /^(csv|html)/ ) {
        $type = "text/$http_format";
    }
    else {
        $type = "application/x-$http_format";
        warn "Unknown mime type $http_format, fall back to $type"
          if $self->debug;
    }
    warn "Set mime type to '$type'\n" if $self->debug;

    $r->content_type($type);

    my $data = "";
    while (<$fh>) {
        $data .= $_;
    }

    my $elapsed = int( tv_interval($t0) * 100 ) / 100;
    warn "$message in $elapsed seconds\n" if $self->debug;

    return $data;
}

1;
