# cd /usr/local/src/git/mkadmin && MAloglevel=torusop,torus,wsput,wspost perl -I lib t/Masterkey-Admin-Torus.t

use strict;
use warnings;
use Test::More tests => 72;
use Data::Dumper; $Data::Dumper::Indent = 1;

use lib qw(lib);

BEGIN {
    use_ok('Masterkey::Admin');
};

chomp(my $pwd = `pwd`);
$ENV{MAconfigpath} ||= "$pwd/../mk2-config-newdemo/admin/mkadmin:$pwd/config/system";
$ENV{SERVER_NAME} ||= "mk2.indexdata.com";
#$ENV{MAloglevel} ||= "torus";

# Here's the script:
# 1. Failed login as consortium administrator
# 2. Successful login as consortium administrator
# 3. Fetch consortium administrator record
# 4. Fetch list of selected targets (a "my" list)
# 5. Fetch list of library administrators (a world list)
# 6. Login as "Mike's Library" administrator
# 7. Fetch list of selected categories (a "my" list)
# 8. Fetch list of end-users (a world list)
# 9. Fetch list of selected targets
# 10. Fetch a specific target
# 11. Modify and update that target
# 12. Revert that modification, removing the override
# 13. Add a library target from scratch
# 14. Delete the added target
# 15. Add a library target based on a consortium target
# 16. Delete the added target
# This sequence of operations mimics pretty much everything that you
# might do in an MKAdmin session.

my($rs, $consortiumUser, $libraryUser, $fields, $id);

# Make toruses for "admin" tab -- used by operations 1-3, 5
my($adminTorus, $adminAuthTorus) = get_toruses("admin");
$adminTorus->time_start();

# 1. Failed login as consortium administrator
$consortiumUser = $adminAuthTorus->user1(password=>"adminx", userName=>"admin");
ok(!defined $consortiumUser, "unsuccessful login search finds no records");

# 2. Successful login as consortium administrator
my($user, $pw) = ("admin", "admin");
$consortiumUser = $adminAuthTorus->user1(password=>$pw, userName=>$user);
ok(defined $consortiumUser, "successful consortium login search finds a record");
$fields = test_record($consortiumUser, "identity", "final");
is($fields->{userName}, $user, "consortium username is correct");
is($fields->{password}, $pw, "consortium password is correct");
$id = $fields->{id};
ok(defined $id, "got id of consortium administrator");
is($id, "master-user.xml", "id of consortium administrator is correct");

# 3. Fetch consortium administrator record
$consortiumUser = $adminAuthTorus->user1(id => $id);
# This die()s if the record isn't found, so no need to test.
$fields = test_record($consortiumUser, "identity", "final");
is($fields->{id}, $id, "consortium administrator has correct id");

# Make Torus object for "global" tab -- used by operation 4
my($globalTorus) = get_toruses("global");

# 4. Fetch list of selected targets (a "my" list)
$rs = $globalTorus->records($consortiumUser->field("identityId"), '', $consortiumUser);
ok($rs->alln() > 1, "global list finds multiple records (" . $rs->alln() . ")");

# 5. Fetch list of library administrators (a world list)
$rs = $adminTorus->records('admin', '', $consortiumUser);
ok($rs->alln() > 1, "library list finds multiple records (" . $rs->alln() . ")");

# Find the "Mike's Library" administrator
$rs = $adminTorus->records('admin', 'userName==mike', $consortiumUser);
is($rs->alln(), 1, "successful library login search finds one record");
$libraryUser = $rs->record(0);
$fields = test_record($libraryUser, "identity", "final");
is($fields->{userName}, "mike", "library username is correct");

# Make torus objects for "searchable" tab -- used by operation 6 and 9-16
my($searchableTorus, $searchableAuthTorus) = get_toruses("searchable");

# 6. Login as "Mike's Library" administrator
$libraryUser = $searchableAuthTorus->user1(id => $fields->{id});
$fields = test_record($libraryUser, "identity", "final");
is($fields->{userName}, "mike", "library username is still correct");
$id = $fields->{id};
$libraryUser = $searchableAuthTorus->user1(id => $id);
$fields = test_record($libraryUser, "identity", "final");
is($fields->{id}, $id, "library administrator has correct id");
my $libraryRealm = $libraryUser->field("identityId");

# Make torus object for "lcat" tab -- used by operation 7
my($lcatTorus) = get_toruses("lcat");

# 7. Fetch list of selected categories (a "my" list)
$rs = $lcatTorus->records($libraryRealm, '', $libraryUser);
ok($rs->alln() > 1, "mike's category list finds multiple records (" . $rs->alln() . ")");

# Make torus object for "identity" tab -- used by operation 8
my($identityTorus) = get_toruses("identity");

# 8. Fetch list of end-users (a world list)
$rs = $identityTorus->records($libraryRealm, '', $libraryUser);
is($rs->alln(), 1, "mike's end-user list finds a single record");

# 9. Fetch list of selected targets
$rs = $searchableTorus->records($libraryRealm, 'science', $libraryUser);
ok($rs->alln() > 1, "mike's searchable list finds multiple science records (" . $rs->alln() . ")");

# For the next few operations, we need a record that has no comment
my $record = undef;
my $n = $rs->alln();
for (my $i = 0; $i < $n; $i++) {
    my $rec = $rs->record($i);
    if (!defined $rec->field("comment", "override")) {
	$record = $rec;
	last;
    }
}
ok(defined $record, "found a record with no comment override (" . $record->displayName() . ")");
die "no records without comment overrides -- can't continue" if !defined $record;
my $recordId = $record->field("id");

# 10. Fetch a specific target
#	record(searchable: realm='mike_s_private_library', id='A.B.A.local-0-0-0-0')
$record = $searchableTorus->recordByArgs(realm => $libraryRealm, id => $recordId);
ok(defined $record, "fetch record with no comment override");
is($record->field("id"), $recordId, "record has correct id");

# 11. Modify and update that target; check
my $text = "Temporary comment";
$searchableTorus->update(0, $libraryRealm, $recordId, undef, {}, comment => $text);
ok(1, "updated record");

$record = $searchableTorus->recordByArgs(realm => $libraryRealm, id => $recordId);
ok(defined $record, "record is still there");
is($record->field("comment", "override"), $text, "comment has been updated");

# 12. Revert that modification, removing the override
$searchableTorus->update(0, $libraryRealm, $recordId, undef, { comment => undef });
ok(1, "reverted record");

$record = $searchableTorus->recordByArgs(realm => $libraryRealm, id => $recordId);
ok(defined $record, "record is still there");
ok(!defined $record->field("comment", "override"), "comment has been removed");

# 13. Add a library target from scratch
$text = "FooBar";
$recordId = $searchableTorus->add(0, $libraryRealm, undef, $libraryUser, displayName => $text);
ok(defined $recordId, "made new record (id=$recordId)");

$record = $searchableTorus->recordByArgs(realm => $libraryRealm, id => $recordId);
ok(defined $record, "new record exists");
is($record->field("displayName"), $text, "new record has correct displayName");

# 14. Delete the added target
$searchableTorus->deleteByArgs(realm => $libraryRealm, id => $recordId);
# No return-value to test
$rs = $searchableTorus->records($libraryRealm, qq[id="$recordId"], $libraryUser);
is($rs->alln(), 0, "deleted record can no longer be found");

# Find a world record
$rs = $searchableTorus->world($libraryRealm, 'science');
ok($rs->alln() > 1, "searchable world finds multiple science records (" . $rs->alln() . ")");
my $worldRecord = $rs->record(0);

# 15. Add a library target based on a consortium target
$recordId = $searchableTorus->add(0, $libraryRealm, $worldRecord->field("id"), $libraryUser);
ok(defined $recordId, "made cloned record (id=$recordId)");

$record = $searchableTorus->recordByArgs(realm => $libraryRealm, id => $recordId);
ok(defined $record, "cloned record exists");
is($record->field("worldId"), $worldRecord->field("id"), "cloned record has correct worldId");
is($record->field("displayName"), $worldRecord->field("displayName"), "cloned record has correct displayName");

# Count matching records; delete the added target, check reduced count
my $rs2 = $searchableTorus->records($libraryRealm, qq[id="$recordId"], $libraryUser);
my $pre_n = $rs2->alln();
$searchableTorus->deleteByArgs(realm => $libraryRealm, id => $recordId);
# No return-value to test
$rs = $searchableTorus->records($libraryRealm, qq[id="$recordId"], $libraryUser);
is($rs->alln(), $pre_n-1, "deleted record can no longer be found (count=" . $rs->alln() . ")");

cmp_ok($adminTorus->time_diff, ">", 3_000, "Tests runs for @{[ $adminTorus->time_diff ]} milliseconds");

my @_get_toruses_keepAlive; # PRIVATE to get_toruses()
sub get_toruses {
    my($tabName) = @_;

    my $admin = new Masterkey::Admin($tabName);
    ok(defined $admin, "made Admin object for '$tabName' tab");
    push @_get_toruses_keepAlive, $admin; # Prevent garbage collection
    my $torus = $admin->torus();
    ok(defined $torus, "got Torus object for '$tabName' tab");
    my $authTorus = $admin->authTorus();
    ok(defined $authTorus, "got auth Torus object for '$tabName' tab");
    return ($torus, $authTorus);
}


sub test_record {
    my($record, $type, $lname) = @_;

    isa_ok($record, "Masterkey::Admin::Record");
    is($record->{type}, $type, "record is of type '$type'");
    my $layer = $record->{layermap}->{$lname};
    ok(defined $layer, "got '$lname' layer");
    my $fields = $layer->{fields};
    ok(defined $fields, "got fields from '$lname' layer");
    return $fields;
}
