package Masterkey::Admin;

use 5.008;
use strict;
use warnings;

use HTML::Entities; # For encode_entities()
use Apache2::Cookie;
use Data::Dumper;

use Masterkey::Admin::Session;
use Masterkey::Admin::Config;
use Masterkey::Admin::Torus;

our @ISA = qw();
our $VERSION = '3.29';

sub new {
    my $class = shift();
    my($tabName) = @_;
    die "$class->new() called with more than one argument: @_" if @_ > 1;

    my $this = bless {
	options => {},
	siteConfig => undef,
	tabName => $tabName,
	config => undef,
	# Use these if they are defined ...
	realm => undef, authRealm => undef,
	# ... Fall back on these otherwise
	torus => undef, authTorus => undef,
	fieldsWithDefaultValues => undef, # until set by first invocation
    }, $class;

    $this->option(logprefix => $0);
    $this->option(loglevel => $ENV{MAloglevel} || 0);

    my @path = $this->_path();
    $this->{siteConfig} = new Masterkey::Admin::Config($this, undef, @path);
    if (defined $tabName) {
	$this->{config} = new Masterkey::Admin::Config($this, $tabName, @path);
	die "old-style Torus1 configuration: 'torus' directive used"
	    if $this->conf("torus");
	if (my $ws = $this->conf("realm")) {
	    $this->{torus} = new Masterkey::Admin::Torus($this, $ws);
	}
	die "old-style Torus1 configuration: 'authTorus' directive used"
	    if $this->conf("authTorus");
	if (my $ws = $this->conf("authRealm")) {
	    $this->{authTorus} = new Masterkey::Admin::Torus($this, $ws);
	    $this->{authTorus}->{isAuth} = 1;
	}
    }

    return $this;
}


sub _path {
    my $this = shift();

    my $path = $ENV{MAconfigpath};
    if ($path) {
	return split(/:/, $path);
    }

    # Continue to support old versions for now
    my $dir = $ENV{MAconfigdir} || die "no MAconfigdir defined in environment";
    my $sysdir = $ENV{MAsysdir} || "/etc/masterkey/admin/include";

    return ($dir, $sysdir);
}


sub siteConfigObject { shift()->{siteConfig} }
sub tabName { shift()->{tabName} }
sub configObject { shift()->{config} }
sub torus { shift()->{torus} }
sub authTorus { shift()->{authTorus} }


sub fieldsWithDefaultValues {
    my $this = shift();

    if (!defined $this->{fieldsWithDefaultValues}) {
	my @candidates = $this->configObject()->configKeys("field.");
	my @actual = grep { /^field\..*\.defaultValue/ } @candidates;
	$this->{fieldsWithDefaultValues} = [ map { s/^field\.(.*)\.defaultValue/$1/; $_ } @actual ];
    }

    return @{ $this->{fieldsWithDefaultValues} };
}


sub new_with_session {
    my $class = shift();
    my($tabName, $r, $nouser) = @_;

    my $admin;
    eval {
	$admin = new Masterkey::Admin($tabName);
    }; if ($@) {
	die "No configuration for tab '$tabName': $@";
    }

    my $domain = $admin->conf("authDomain");
    my $session = undef;
    my $cookies = Apache2::Cookie->fetch($r);
    my $cookie = $cookies->{"session-$domain"};
    if (defined $cookie) {
	my $cval = $cookie->value();
	$session = Masterkey::Admin::Session->find($cval);
	$admin->log(1, "expiring old session $cval") if !defined $session;
    }

    if (!defined $session) {
	$session = Masterkey::Admin::Session->new();
	my $cookie = new Apache2::Cookie($r, -path => "/",
					 -name => "session-$domain",
					 -value => $session->cookie());
	$cookie->bake($r);
    }

    my $user = undef;
    goto DONE if $nouser;

    my $uid = $session->user_id();

    # On http://ohiolink-test.indexdata.com/ this fails randomly
    # about half the time, saying "Undefined subroutine
    # &Apache2::Request called at
    # /usr/lib/perl5/site_perl/5.8.8/Masterkey/Admin.pm line 115".
    # We don't understand why, so for now we just comment it out.
#    if (!$uid) {
#	# Attempt an immediate login with supplied parameters, if any
#	my $usefulR = new Apache2::Request($r);
#	my $user = $usefulR->param("authUser");
#	my $password = $usefulR->param("authPassword");
#	if (defined $user && defined $password) {
#	    warn "### we should log in as $user/$password";
#	}
#    }

    if (!$uid) {
	# Check whether we were logged into the Harvester Console,
	# which sets authUserCookie, and if so, trust that we can
	# adopt the same login here.  Note that ### THIS IS A HUGE
	# SECURITY HOLE.
	# I feel bad that this is in the library but the complementary
	# code that sets this cookie is in a Mason component,
	# login/doit.mc.  Probably that should pulled down into this
	# library.  Oh well.
	my $cname = $admin->conf("authUserCookie");
	if (defined $cname) {
	    my $cookie = $cookies->{$cname};
	    if (defined $cookie) {
		$uid = $cookie->value();
	    }
	}
    }

    if ($uid) {
	eval {
	    $user = $admin->authTorus()->user1(id => $uid);
	}; if ($@) {
	    if (ref($@) eq "HTML::Mason::Exception" && $@ =~ 404) {
		# Assume the record has been deleted
		goto DONE;
	    } else {
		die $@;
	    }
	}
	die "Invalid user-ID '$uid'" if !defined $user;

	# Special case: consortium-level administrator is always super
	$user->{layers}->{final}->{fields}->{superuser} = 1
	    if $domain eq "wizard";
    }

  DONE:
    return ($admin, $session, $user);
}

sub siteConf {
    my $this = shift();
    my($key) = @_;

    return $this->siteConfigObject()->value($key);
}

sub configKeys {
    my $this = shift();
    my($prefix) = @_;

    return $this->configObject()->configKeys($prefix);
}

sub conf {
    my $this = shift();
    my($key) = @_;

    my $config = $this->configObject();
    return $config->value($key) if defined $config;
    return $this->siteConf($key);
}

# Returns a caption to be used in the Web UI for the named element,
# defaulting to the specified value if no such caption is defined in
# the per-tab configuration.  Note that the value is quoted ready for
# insertion into HTML.
#
sub caption {
    my $this = shift();
    my($key, $default) = @_;

    my $val = $this->conf("caption.$key");
    return encode_entities($val) if defined $val;
    return encode_entities($default);
}

sub option {
    my $this = shift();
    my($key, $value) = @_;

    my $old = $this->{options}->{$key};
    if (defined $value) {
	# Special cases for "loglevel" to allow hex and octal bitmasks
	# and to parse non-numeric level-lists.
	if ($key eq "loglevel") {
	    $value = oct($value)
		if $value =~ /^0/;
	    $value = Masterkey::Admin::LogLevel::num($value)
		if $value !~ /^\d+$/;
	}
	#print STDERR "setting '$key' to '$value'\n";
	$this->{options}->{$key} = $value;
    }

    return $old;
}

sub log {
    my $this = shift();
    my($level, @args) = @_;

    if ($this->option("loglevel") & $level) {
	### could check another option for whether to include PID
	my $label = Masterkey::Admin::LogLevel::label($level);
	print STDERR $this->option("logprefix"), " ($label): ", @args, "\n";
    }
}

# Returns human-readable form of named field
sub fieldname {
    my $this = shift();
    my($context, $name) = @_;

    my $val;
    if (defined $name) {
	# Two-argument version that defines a context
	$val = $this->conf("$context.field.$name.name");
    } else {
	# Single-argument version that defines only the name
	$name = $context;
	$val = $this->conf("field.$name.name");
    }

    return $val if defined $val;

    # Otherwise, return a non-mangled version for completely unknown
    # fields, as these will often have been added by hand: see
    # MKA-108.
    return $name if !$this->conf("field.$name._exists");

    $name =~ s/([A-Z])/ $1/g;
    return ucfirst($name);
}


# Returns undef if any mandatory fields of the new records are missing,
# otherwise the ID of the new record.
#
sub make_record {
    my $this = shift();
    my($identity, $user, $master, %data) = @_;

    my $torus = $this->torus();
    my @fields = $this->configObject()->profile()->fields();

    if (!$master) {
	foreach my $ref (@fields) {
	    my($name, $mandatory) = @$ref;
	    my $val = $data{$name};
	    if ($mandatory && (!defined $val || $val eq "")) {
		return undef;
	    }
	}
    }

    my $id = $torus->add($master, $identity, undef, $user, %data);

    my($caption, $rendered);
    if ($master) {
	$caption = 'master record';
	$rendered = '{' . join(', ', map { $_ . '="' . $data{$_} . '"' } sort keys %data) . '}';
    } else {
	$caption = 'record';
	$rendered = '"' . $data{displayName} . '"';
    }
    warn "added $caption $rendered";

    if ($this->conf("email.event.newRecord") ||
	$this->conf("sendEmail.new")) {
	$this->send_email(($this->conf("email.toAdmin") ||
			   $this->conf("adminEmail")),
			  ("New $caption added"),
			  "User '" . $user->displayName() . "' " .
			  "added record " . $rendered . " (id=$id)");
    }

    $torus->make_realms_for_record(%data);

    return $id;
}


# Returns undef if any mandatory fields of the record are missing,
# otherwise the record that was used in checking the presence of
# fields.  This records has to be obtained in order to do the
# validation: it's returned here so that the caller doesn't need to
# ask for it again.
#
sub modify_record {
    my $this = shift();
    my($identity, $id, undef, $goneHash, %data) = @_;

    $this->torus()->update($identity, $id, undef, $goneHash, %data);
    return undef;
}


sub send_email {
    use Net::SMTP;

    my $this = shift();
    my($to, $subject, $text) = @_;

    my $from = $this->conf("email.from") ||
	'mike@indexdata.com (MasterKey Admin UI)';
    my $mailHost = $this->conf("email.host") ||
	"localhost";
    my $mailHello = $this->conf("email.hello") ||
	"mk2.indexdata.com";
    # Could allow configuration of user/password if needed

    my $data = "From: $from\nTo: $to\nSubject: $subject\n\n$text";
    my $smtp = new Net::SMTP($mailHost, Hello => $mailHello)
	or die "can't make SMTP object";
    $smtp->mail($from) or die "can't send email from $from";
    $smtp->to($to) or die "can't use SMTP recipient '$to'";
    $smtp->data($data) or die "can't email data to '$to'";
    $smtp->quit() or die "can't send email to '$to'";
}

sub dump {
    my $this = shift();
    local $Data::Dumper::Indent = 1;
    return Dumper(@_);
}


# The Torus often returns rather verbose error messages which we don't
# need to show to the user in detail. This function trims them of the
# extraneous detail, retaining information only if $debug is set.
#
sub exception2error {
    my $this = shift();
    my($s, $debug) = @_;

    $s =~ s/Stack:.*//s if !$debug;
    $s =~ s/.*Parsing query failed -(.*) at character \d+\)/Bad query:$1/s;
    return qq[   <p class="error">$s</p>\n];
}

1;
